/*
 *
 *                       ADOBE CONFIDENTIAL
 *                     _ _ _ _ _ _ _ _ _ _ _ _
 *
 * Copyright 2006, Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any.  The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law.  Dissemination of this information or reproduction of this material is
 * strictly forbidden unless prior written permission is obtained from Adobe
 * Systems Incorporated.
 *
 * Author: Varahamurthy Jutur, 09-JAN-2014
 */

/*jslint plusplus: true white: true*/
/*global window, AdobeRDMHelper, readium, EPUBcfi, console, external*/
/*global AdobeCfiIterator, AdobeCfiTextIterator*/
/*global jQuery, $*/
// FILE START: AdobeRDMHelper.js
(function () {
  'use strict';

  var SearchHelper,
    Utility,
    Tracer;

  /**Some helpers for search functionality*/
  SearchHelper = {
    /**Keep these in sync with rmsdk_wrapper/public/rmsdk_types.h*/
    Flags: {
      /**< Match the case of characters in the string being searched. */
      SF_MATCH_CASE: 1,
       /**< Search toward the beginning of the document. */
      SF_BACK: 2,
      /**< Match whole word only. */
      SF_WHOLE_WORD: 4,
      /**< Wrap search: start from the beginning when the end is reached. */
      SF_WRAP: 8,
      /**< Ignore all accents on latin characters. SF_MATCH_CASE and SF_IGNORE_ACCENTS may be used in any combination. */
      SF_IGNORE_ACCENTS: 0x10
    },

    decodeFlags: function (flags) {
      /*jslint bitwise: true*/
      var flagObj = {};

      if (flags) {
        if (flags & this.Flags.SF_MATCH_CASE) {
          flagObj.caseSensitive = true;
        }
        if (flags & this.Flags.SF_BACK) {
          flagObj.reverse = true;
        }
        if (flags & this.Flags.SF_WHOLE_WORD) {
          flagObj.matchWord = true;
        }
        if (flags & this.Flags.SF_WRAP) {
          flagObj.shouldWrap = true;
        }
        if (flags & this.Flags.SF_IGNORE_ACCENTS) {
          flagObj.ignoreAccents = true;
        }
      }

      return flagObj;
    }
  };

  /**@public Utility to track errors through logs*/
  Tracer = function (where) {
      this.where = where || "Tracer";
      this.messages = [];
  };
  Tracer.prototype.log = function (msg) {
      this.messages.push(msg);
  };
  Tracer.prototype.trace = function () {
      return '[In ' + this.where + ']\n' + this.messages.join('\n');
  };

  /**Some utility functions used by AdobeRDMHelper
   */
  Utility = {};
  Utility.Tracer = Tracer;

  /**Escape special characters so the string is used as such
   * The logic is clear - this link specifies how order is important, which
   * I (shakrish) have not tested so including link here...
   * http://stackoverflow.com/questions/3446170/escape-string-for-use-in-javascript-regex
   */
  Utility.escapeRegexString = function (str) {
    /**Long Form
     * var specials = ['-','[',']','/','{','}','(',')','*','+','?','.','\\','^','$','|'];
     * var regex = new RegExp('[' + specials.join('\\') + ']', 'g'); //Search for all occurences...
     * //Replace with a '\' followed by (the match $&)[https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/replace#Description]
     * return str.replace(regex, '\\$&'); 
     */

    /**Short Form*/
    return str.replace(/[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, "\\$&");
  };

  Utility.createRectFromPoints = function (coordinates) {
    var rect = {
      top: coordinates.top || 0,
      bottom: coordinates.bottom || 0,
      left: coordinates.left || 0,
      right: coordinates.right || 0
    };
    rect.height = rect.bottom - rect.top;
    rect.width = rect.right - rect.left;

    return rect;
  };

  /**getClientRect() and getBoundingClientRect() return read-only data*/
  Utility.createRectFromClientRect = function (clientRect) {
    return Utility.createRectFromPoints({
      top: clientRect.top,
      bottom: clientRect.bottom,
      left: clientRect.left,
      right: clientRect.right
    });
  };

  /**See cfi_navigation_logic > checkVisibilityByRectangles
   * The bounding rect returned may or may not be split up across pages
   * So we normalize it by spliting in any case*/
  Utility.splitRectAcrossColumns = function (clientRect, viewportDimensions) {
    var splitRects = [],
      eachRect,
      rect = Utility.createRectFromClientRect(clientRect);

    /**If rect overflows across columns,
      * split it up manually using viewportDimensions*/
    while (rect.bottom > viewportDimensions.height) {
      eachRect = Utility.createRectFromPoints({
        top: rect.top,
        bottom: viewportDimensions.height,
        left: rect.left,
        right: rect.right
      });

      splitRects.push(eachRect);

      //'Move' to the next rectangle...
      rect = Utility.createRectFromPoints({
        top: 0,
        bottom: rect.bottom - viewportDimensions.height,
        left: rect.left + viewportDimensions.width,
        right: rect.right + viewportDimensions.width
      });
    }
    splitRects.push(rect);

    return splitRects;
  };

  Utility.rectInRect = function (childRect, parentRect) {
    return (childRect.top >= parentRect.top && childRect.bottom <= parentRect.bottom
      && childRect.left >= parentRect.left && childRect.right <= parentRect.right);
  };

  Utility.getSplitClientRects = function (clientRects, viewportDimensions) {
    var actualRects = [],
      height = 0,
      partialResult,
      partialLen,
      i,
      j;
    for (i = 0; i < clientRects.length; ++i) {
      partialResult = Utility.splitRectAcrossColumns(clientRects[i], viewportDimensions);
      partialLen = partialResult.length;
      for (j = 0; j < partialLen; ++j) {
        height += partialResult[j].height;
        actualRects.push(partialResult[j]);
      }
    }
    return {
      rects: actualRects,
      totalHeight: height
    };
  };

  Utility.getRectOffsetsInParent = function (clientRects, offsetRect, viewportDimensions) {
    var parentHeightSeen = 0,
      parentHeightTotal = 0,
      result = {
        totalHeight: 1, //To avoid 0/0 = NaN by default
        offsetToRect: 0
      },
      actualRects = [],
      len,
      eachRect,
      actualInfo,
      i;

    actualInfo = Utility.getSplitClientRects(clientRects, viewportDimensions);
    actualRects = actualInfo.rects;
    len = actualRects.length;
    for (i = 0; i < len; ++i) {
      eachRect = actualRects[i];
      if (Utility.rectInRect(offsetRect, eachRect)) {
        /**Boundary Case Bug: <see trimRectanglesByVertOffset>
         * Since we take floor of this seen value after division
         * it causes the first line top to sometimes be calculated as a
         * small negative value. This causes it to be interpreted as last line
         * of previous page. To fix this, we'll return the midpoint of the line */
        parentHeightSeen = parentHeightTotal + offsetRect.top - eachRect.top + (offsetRect.height / 2);
      }
      parentHeightTotal += eachRect.height;
    }

    result.totalHeight = parentHeightTotal;
    result.offsetToRect = parentHeightSeen;
    return result;
  };

  Utility.scaleHeightOffset = function (parentNode, normalizedHeight) {
    var totalHeight = 0,
      offsetHeight,
      i,
      clientRects;

    clientRects = parentNode.getClientRects();
    for (i = 0; i < clientRects.length; ++i) {
      totalHeight += clientRects[i].height;
    }
    if (normalizedHeight === undefined || normalizedHeight < 0) {
      normalizedHeight = 0;
    } else if (normalizedHeight > 100) {
      normalizedHeight = 100;
    }

    offsetHeight = Math.ceil(totalHeight * normalizedHeight / 100);
    return offsetHeight;
  };

  Utility.getTextNodeClientRects = function (textNode, contentDoc) {
    /*global rangy*/
    var USE_RANGY = true,// && (rangy !== undefined),
      offsetRects = [],
      range,
      nRange;

    if (USE_RANGY) {
      range = rangy.createRange(contentDoc);
      range.selectNode(textNode);

      nRange = (range.nativeRange || range);
      offsetRects = nRange.getClientRects();
    }
    return offsetRects;
  };

  /**Searching through child client rects is non trivial
   * since each client rect may or may not lie on a different line
   * and may or may not flow across pages
   * We can assume the rects are always given in order */
  Utility.getHeightOfRectList = function (rectList) {
    var i,
      len = rectList.length,
      height,
      topBound,
      bottomBound,
      eachRect;

    i = topBound = bottomBound = height = 0;
    while (i < len) {
      //Reset bounds to this rect...
      eachRect = rectList[i];
      topBound = eachRect.top;
      bottomBound = eachRect.bottom;

      //Continue through the rects...
      while (++i < len) {
        eachRect = rectList[i];
        if (topBound > eachRect.top) {
          //We've flowed into a new page layout. Break.
          break;
        }
        bottomBound = Math.max(bottomBound, eachRect.bottom);
      }

      //Add the height seen so far...
      height += bottomBound - topBound;
    }
    return height;
  };

  /**Opposite of split client rects. Returns the merged bounding rect
   * firstRect and secondRect are passed in order of occurence*/
  Utility.mergeRectsAcrossPages = function (firstRect, secondRect, viewportDimensions) {
    var result;

    if (secondRect.top < firstRect.top) {
      //Overflow
      result = Utility.createRectFromPoints({
        top: firstRect.top,
        bottom: firstRect.bottom + secondRect.height,
        left: Math.min(firstRect.left, secondRect.left - viewportDimensions.width),
        right: viewportDimensions.width
      });
    } else {
      //Not an overflow...
      result = Utility.createRectFromPoints({
        top: firstRect.top,
        bottom: secondRect.bottom,
        left: Math.min(firstRect.left, secondRect.left),
        right: Math.max(firstRect.right, secondRect.right)
      });
    }

    return result;
  };

  /**Opposite of split client rects. Returns the merged bounding rect*/
  Utility.getMergedClientRect = function (clientRects, viewportDimensions) {
    var i,
      boundRect;

    if (clientRects.length) {
      boundRect = Utility.createRectFromClientRect(clientRects[0]);
    } else {
      boundRect = Utility.createRectFromPoints({ top: 0, bottom: 0, left: 0, right: 0});
    }
    for (i = 1; i < clientRects.length; ++i) {
      boundRect = Utility.mergeRectsAcrossPages(boundRect, clientRects[i], viewportDimensions);
    }

    return boundRect;
  };

  Utility.getChildAtOffsetV = function (parentNode, scaledVerticalOffset, contentDoc, viewportDimensions) {
    var childCount = parentNode.childNodes.length,
      eachChild,
      clientRects,
      i,
      result = {
        node: null,
        relativeOffset: 0
      },
      childRect,
      childUnseenHeight,
      seenBottom = 0;

    /*global Node*/
    for (i = 0; i < childCount; ++i) {
      eachChild = parentNode.childNodes[i];
      if (eachChild.nodeType === Node.TEXT_NODE) {
        clientRects = Utility.getTextNodeClientRects(eachChild, contentDoc);
      } else {
        clientRects = eachChild.getClientRects();
      }

      childRect = Utility.getMergedClientRect(clientRects, viewportDimensions);
      if (i === 0) {
        seenBottom = childRect.top;
      }
      childUnseenHeight = childRect.bottom - seenBottom;
      if (childUnseenHeight > 0) {
        //We've not seen at least part of this child's height...
        seenBottom = childRect.bottom;
        if (scaledVerticalOffset <= childUnseenHeight) {
          //Using break so the matched i is available debugging
          break;
        }
        /**SeenBottom = coordinate system origin
         * Transform offset to this system*/
        scaledVerticalOffset -= childUnseenHeight;
      }
    }
    if (i < childCount) {
      result.node = eachChild;
      result.relativeOffset = scaledVerticalOffset;
    }
    return result;
  };

  Utility.offsetVInRectList = function (verticalOffset, rectList) {
    var found = false,
      i;
    for (i = 0; i < rectList.length; ++i) {
      if (verticalOffset <= rectList[i].height) {
        found = true;
        break;
      }
      verticalOffset -= rectList[i].height;
    }
    return found;
  };

  Utility.getCharacterOffsetAtOffsetV = function (textNode, scaledVerticalOffset, contentDoc) {
    var offset = 0,
      USE_RANGY = true,// && (rangy !== undefined),
      range,
      nRange,
      startOffset = 0,
      // mid,
      endOffset = textNode.length - 1,
      offsetRects,
      lowerBound,
      upperBound,
      guessEnd;

    //We have a text node child... now find the offset internally...
    if (USE_RANGY) {
      range = rangy.createRange(contentDoc);
      nRange = (range.nativeRange || range);
      range.setStart(textNode, startOffset);

      //Now, create trial-and-error ranges to get one that falls in this...
      lowerBound = startOffset;
      upperBound = endOffset;
      do {
        guessEnd = Math.floor((lowerBound + upperBound) / 2);
        range.setEnd(textNode, guessEnd);

        offsetRects = nRange.getClientRects();
        if (Utility.offsetVInRectList(scaledVerticalOffset, offsetRects)) {
          upperBound = guessEnd;
        } else {
          lowerBound = guessEnd + 1;
        }
      } while (lowerBound < upperBound);
      offset = upperBound;
    }
    return offset;
  };

  /**Convert a spatial CFI to the correct element + offset
   * This Method WILL ONLY WORK for a content Doc that is displayed on the screen
   * @param {String} elementCFI CFI String to process
   * @param @optional {Document} contentDoc Document context to run this in. Defaults to first active frame's document
   * References:
   * http://stackoverflow.com/questions/9345275/range-from-pixels
   * http://stackoverflow.com/questions/11225933/select-from-pixel-coordinates-for-ff-and-google-chrome/11336426#11336426
   * https://code.google.com/p/rangy/wiki/RangyRange
   * http://www.zehnet.de/2010/11/19/document-elementfrompoint-a-jquery-solution/
   * http://stackoverflow.com/questions/13597157/get-dom-text-node-from-point
   */
  Utility.convertSpatialCFI = function (parent, offsetX, offsetY) {
    var result = {
        node: null,
        CFI: ''
      },
      offsets,
      nodeInfo;

    offsets = {
      x: offsetX,
      y: offsetY,
      scaled: false
    };

    nodeInfo = AdobeRDMHelper.getChildNodeFromSpatialOffsets(parent, offsets);
    if (nodeInfo.offset !== undefined) {
      result.CFI = AdobeRDMHelper.generateTextElementCFI(nodeInfo.node, nodeInfo.offset);
    } else {
      result.CFI = AdobeRDMHelper.generateElementCFI(nodeInfo.node);
    }
    result.node = nodeInfo.node;

    return result;
  };
  Utility.stripSpatialCFI = function (elementCFI) {
    var result = {
        CFI: elementCFI,
        x: 0,
        y: 0,
        foundSpatial: false
      },
      index,
      offsetString;

    index = elementCFI.indexOf('@');
    if (index > -1) {
      offsetString = elementCFI.substr(index + 1);
      result.CFI = elementCFI.substr(0, index);

      index = offsetString.indexOf(':');
      result.x = parseInt(offsetString.substr(0, index), 10);
      result.y = parseInt(offsetString.substr(index + 1), 10);
      result.foundSpatial = true;
    }
    return result;
  };

  Utility.getCharacterOffsetFromCFI = function (CFI) {
    CFI = CFI || '';
    var index,
      offset = 0,
      offsetStr = '';

    index = CFI.lastIndexOf(':');
    if (index > -1) {
      offsetStr = CFI.substr(index + 1);
      offset = parseInt(offsetStr, 10);
    }
    return offset;
  };

  /**Replaces all white spaces with a ' ' character
   * Treats consecutive white spaces as a single block
   * @param {String} text
   * @param {String} index
   * @return {Object} text index Updated values*/ 
  Utility.stripWhiteSpacesAndAdjustOffset = function (text, index) {
    var result = {
      text: '',
      index: index
    },
    matchSource = text,
    matchIndex,
    matchString,
    regexResult;

    regexResult = matchSource.match(/\s+/);
    while(regexResult) {
      /**Get match data*/
      matchIndex = regexResult.index;
      matchString = regexResult[0];
           
      if (result.text.length + matchIndex < result.index) {
        /**Splice modifies index, update */
        result.index -= matchString.length - 1;

        /**Store the data seen so far and splice the rest*/
        result.text += matchSource.substr(0, matchIndex) + ' ';
        matchSource = matchSource.substr(matchIndex + matchString.length);
        
        /**Move to next match*/
        regexResult = matchSource.match(/\s+/);
      } else {
        /*Index crossed - splice away to heart's content...*/
        matchSource = matchSource.replace(/\s+/g, ' ');
        regexResult = undefined;
      }
    }
    result.text += matchSource;
    
    return result;
  };

// $.getScript("./js/rmsdk_epubReadingSystem.js", function(){
//             console.log("rmsdk_epubReadingSystem.js loaded");
//             // Use anything defined in the loaded script...
//             });
 
    jQuery.ajax({
             async:false,
             type:'GET',
             url:"./rmsdk_epubReadingSystem.js",
             data:null,
             success:function(){ return; },
             dataType:'script',
             error: function(xhr, textStatus, errorThrown) {
                /*global alert*/
                alert("Could not load rmsdk_epubReadingSystem.js. Make sure it is present along with AdobeRDMHelper");
                console.log("rmsdk_epubReadingSystem.js load error " + xhr + " " + textStatus + " " + errorThrown);
                // Look at the `textStatus` and/or `errorThrown` properties.
             }
    });
  /**
   * Top level AdobeRDMHelper namespace
   * @class AdobeRDMHelper
   * @static
   */
  window.AdobeRDMHelper = {

    reader: undefined,
    epubReadingSystem_name: window.rmsdk.name,
    epubReadingSystem_version: window.rmsdk.version,

    /**
     Function to determine from RMSDK side if this js is loaded or not.
     Always returns "yes". Do not modify this.
     @method isLoaded
     @static
     @return {string} "yes"
     */
    isLoaded: function() {
      //console.log("AdobeRDMHelper.isLoaded");
      /*global ReadiumSDK, navigator*/
      if ((ReadiumSDK !== undefined) && (ReadiumSDK.reader !== undefined)) {
        // set this variable USE_ADOBE_DRM to identify if RMSDK is loading this readium SDK
        ReadiumSDK.USE_ADOBE_DRM = 1;
        ReadiumSDK._iOS = navigator.userAgent.match(/(iPad|iPhone|iPod)/g) ? true : false;
        ReadiumSDK._Android = navigator.userAgent.toLowerCase().indexOf('android') > -1;
        ReadiumSDK._Mac = navigator.platform.toLowerCase().indexOf('mac') > -1;
        ReadiumSDK._Win = navigator.platform.toLowerCase().indexOf('win') > -1;
        ReadiumSDK._isMobile = ReadiumSDK._iOS || ReadiumSDK._Android;
        ReadiumSDK._isIEBrowser = AdobeRDMHelper.isIEBrowser();
        return "yes";
      }
      return "no";
    },

    isIEBrowser: function(){
        var ua = window.navigator.userAgent;
        var msie = ua.indexOf('MSIE ');
        var trident = ua.indexOf('Trident/');

        if (msie > 0) {
            // IE 10 or older => return version number
            return parseInt(ua.substring(msie + 5, ua.indexOf('.', msie)), 10);
        }

        if (trident > 0) {
            // IE 11 (or newer) => return version number
            var rv = ua.indexOf('rv:');
            return parseInt(ua.substring(rv + 3, ua.indexOf('.', rv)), 10);
        }

        // other browser
        return false;      
    },

    Utility: Utility,

    hostLog: function(msg){
        AdobeRDMHelper.getReader().trigger(ReadiumSDK.Events.GENERIC_HOST_CALLBACK, {msgType:"logMessage", msgData:msg});
    },

    redirectLogsToHost: function(){
      console.log = console.error = AdobeRDMHelper.hostLog;
    },

    /**
     Returns CFI for the given element id. The element should be already loaded in the page when 
     this method is called
     @method getCFIForElementName
     @static
     @return {string} the epub CFI of the given element id
     */
    getCFIForElementName: function(elementId) {
      /*jslint vars: true white: true*/
      var loadedSpineItems = this.getReader().getLoadedSpineItems();
      var cnt = loadedSpineItems.length;
      //this.getReader().trigger(ReadiumSDK.Events.CONSOLE_LOG, cnt);
      var __el, i;
      for (i=0;i<cnt;i++) {
          __el = this.getReader().getElement(loadedSpineItems[i], elementId);
          if(__el !== undefined) {
            break;
          }
      }
      return EPUBcfi.Generator.generateElementCFIComponent(__el);
    },

    isPlayingMediaOverlay: function(){
      try{
        return JSON.stringify(this.getReader().isPlayingMediaOverlay());
      }catch(e){
        console.log("AdobeRDMHelper.isPlayingMediaOverlay(): " + e);
      }
      return JSON.stringify(false);
    },

    /**
     Function to be called on android to get the return value of jsString after evaluation
     @method evalJsForAndroid
     @static
     @return
     */
    evalJsForAndroid: function(evalJs_index, jsString) {
      /*jslint vars: true white: true*/
      console.log("evalJsForAndroid: arg1:" + evalJs_index + " arg2: " + jsString);
      var evalJs_result = "";
      try {
        /*jslint evil: true*/
        evalJs_result = eval(jsString).toString();
      } catch (e) {
          console.log(e);
      }
      window.LauncherUI.setJSRetVal(evalJs_index, evalJs_result);
    },

    /**
     Function to be called on windows to get the return value of jsString after evaluation
     @method evalJsForAndroid
     @static
     @return
     */
    evalJsForWindows: function(jsString) {
      console.log("evalJsForWindows: arg1:" + jsString);
      var evalJs_result = "";
      try {
        /*jslint evil: true*/
        evalJs_result = eval(jsString).toString();
      } catch (e) {
        console.log(e);
      }
      external.setJSRetVal(evalJs_result);
    },
    
    setReader: function(reader){
        this.reader = reader;
    },

    getReader: function() {
      if (this.reader) {
        return this.reader;
      }
      
      if(ReadiumSDK.reader)
      {
        return ReadiumSDK.reader;
      }

      if(readium && readium.reader) // This is for cloud reader 
      {
        return readium.reader;
      }
    },
    
    getSpineItemFromIdref: function(idref) {
        var i,
          spineItems = this.getReader().getLoadedSpineItems(),
          item;
        for(i=0;i<spineItems.length;i++) {
            if( spineItems[i].idref === idref) {
                item = spineItems[i];
            }
        }

        //Couldn't find spine item in loaded list... do a linear search on the spine...
        if (!item && spineItems.length) {
          item = spineItems[0].spine.getItemById(idref);
        }

        return item;
    },
    
    getOffsetFromCFI: function(cfi){
        if(cfi)
        {
            var colonIndex = cfi.lastIndexOf(":");
            if( colonIndex >= 0 ) {
              return parseInt(cfi.substring(colonIndex+1, cfi.length), 10);
            }
        }
        return -1;
    },

    getFirstNonEmptyTextNodeIndex: function($nodeList){
      var i,
        length = $nodeList.length,
        node;
      // getElementByCfi() sometimes returns some empty nodes. We just ignore them here.
      for(i = 0; i < length; i++){
        node = $nodeList[i];
        //Negation of (node.nodeType === Node.TEXT_NODE && (node.data.length === 0 || /^\s*$/.test(node.data)))
        if (node.nodeType !== Node.TEXT_NODE || (node.data.length !== 0 && !/^\s*$/.test(node.data))) {
          break;
        }
      }
      i = (i === length)?0:i;
      //node = ($nodeList && length > 0) ? $nodeList[i] : undefined;
      return i;
    },
    
    getElementByCfi: function(spineItem, partialCFI) {
      /*jslint vars: true white: true*/
      var tracer = new Utility.Tracer('getElementByCfi'),
        $element,
        index,
        result;
      try {
        tracer.log(spineItem.idref + ',' + partialCFI);

        $element = this.getReader().getElementByCfi(spineItem, partialCFI,
                                                  ["cfi-marker", "mo-cfi-highlight", "MathJax_Preview", "MathJax_SVG_Display"],
                                                  [],
                                                  ["MathJax_Message"]);
        tracer.log('Got ' + $element.length + ' matches');
        
        // getElementByCfi() sometimes returns some empty nodes. We just ignore them here.
        index = this.getFirstNonEmptyTextNodeIndex($element);
        tracer.log('Ignored empty nodes');
        
        result = $element[index];
      } catch (error) {
        tracer.log (error.name + ':' + error.message);
        throw new Error (tracer.trace());
      }
      return result;
    },

    getElementByExpandedCfi: function (partialCFI, contentDoc) {
      /*jslint vars: true white: true*/
      var info = {
        element: null,
        CFI: '',
        reloadCFI: false
      };
      
      //Adding support for some special partial CFIs we allow...
      if (partialCFI.indexOf('#') === 0) {
        //CFIs of form <SpineIDRef>!<ElementId> [found in TOC locations]
        info.reloadCFI = true;
        info.element = contentDoc.getElementById(partialCFI.substr(1));
        info.CFI = AdobeRDMHelper.generateElementCFI(info.element);
      } else if (partialCFI === '') {
        //CFIs of form <SpineIDRef>! [found in TOC locations]
        info.element = contentDoc.getElementsByTagName('body')[0].firstChild;
        if (info.element.nodeType === Node.TEXT_NODE) {
          info.CFI = AdobeRDMHelper.generateTextElementCFI(info.element);
        } else {
          info.CFI = AdobeRDMHelper.generateElementCFI(info.element);
        }
        info.reloadCFI = true;
      } else {
        var $element = EPUBcfi.getTargetElementWithPartialCFI("epubcfi(" + partialCFI + ")",
                  contentDoc,
                  ["cfi-marker", "mo-cfi-highlight"],
                  [],
                  ["MathJax_Message"]
                );
        // getTargetElementWithPartialCFI() sometimes returns some empty nodes. We just ignore them here.      
        var index = this.getFirstNonEmptyTextNodeIndex($element);
        
        info.CFI = partialCFI;
        info.element = $element[index];
      }

      return info;
    },

    getElementByPartialCfi: function (partialCFI, contentDoc) {
      /*jslint vars: true white: true*/
      var $element = EPUBcfi.getTargetElementWithPartialCFI("epubcfi(" + partialCFI + ")",
                contentDoc,
                ["cfi-marker", "mo-cfi-highlight", "MathJax_Preview", "MathJax_SVG_Display"],
                [],
                ["MathJax_Message"]
              ),
        index;

      // getTargetElementWithPartialCFI() sometimes returns some empty nodes. We just ignore them here.      
      index = this.getFirstNonEmptyTextNodeIndex($element);
      return $element[index];
    },

    generateElementCFI: function (element) {
      var CFI = EPUBcfi.generateElementCFIComponent(element,
        ["cfi-marker", "mo-cfi-highlight", "MathJax_Preview", "MathJax_SVG_Display"],
        [],
        ["MathJax_Message"]);
      return CFI;
    },

    generateTextElementCFI: function (element, offset) {
      var CFI = EPUBcfi.generateCharacterOffsetCFIComponent(element, (offset || 0),
        ["cfi-marker", "mo-cfi-highlight", "MathJax_Preview", "MathJax_SVG_Display"],
        [],
        ["MathJax_Message"]);
      return CFI;
    },

    getFrames: function (idref) {
      //return $("#epubContentIframe");
      return this.getReader().getContentIFrame(idref);
    },

    getCurrentDocument: function (idref) {
      var frame = this.getFrames(idref)[0];
      return frame.contentDocument;
    },

    getDocumentWindow: function (idref) {
      var frame = this.getFrames(idref)[0];
      return frame.contentWindow;
    },

    /** Convert text offsets to spatial offsets so Readium can understand them
     * See reader_view.js */
    openSpineItemElementCfi: function(idref, elementCfi, initiator) {
      /*jslint unparam: true*/
      var reader = this.getReader(),
        processedCFI = elementCfi;

//    var spineItem = this.getSpineItemFromIdref(idref),
        // index,
        // parentNode,
        // textNode,
        // characterOffset = 0,
        // parentCFI,
        // elementCfiParts,
        // offsets = {};
//        try {
//          //Check if the CFI contains a character offset...
//          index = elementCfi.indexOf(':');
//          if (index > -1) {
//            //Process the text node before modifying anything...
//            textNode = this.getElementByCfi(spineItem, elementCfi);
//            characterOffset = parseInt(elementCfi.substr(index + 1), 10);
//            if (isNaN(characterOffset)) {
//              throw new Error('AdobeRDMHelper: Weird CFI passed to openSpineItemElementCfi()');
//            }
//            
//            //If this offset is split across text nodes...
//            while (textNode && characterOffset >= textNode.nodeValue.length) {
//              characterOffset -= textNode.nodeValue.length;
//              /**Find the next text node... 
//               * Ignore any non-text nodes. Since we assume CFI is valid they're probably 'blacklisted' ones...
//               */
//               do {
//                textNode = textNode.nextSibling;
//               } while (textNode && textNode.nodeType !== Node.TEXT_NODE);
//            }
//            if (!textNode) {
//              throw new Error('AdobeRDMHelper: Invalid CFI passed to openSpineItemElementCfi()');
//            }
//
//            //We need at least the parent and the text node...
//            elementCfiParts = elementCfi.split('/');
//            if (elementCfiParts.length < 2) {
//              throw new Error('AdobeRDMHelper: Weird or Incomplete CFI passed to openSpineItemElementCfi()');
//            }
//            elementCfiParts.pop(); //Remove the last textNode cfi...
//
//            //Process the parent CFI...
//            parentCFI = elementCfiParts.join('/');
//            parentNode = this.getElementByCfi(spineItem, parentCFI);
//
//            offsets = this.convertTextOffsetsToSpatialOffsets(parentNode, textNode, characterOffset);
//            processedCFI = parentCFI + '@' + (offsets.x || '0') + ':' + (offsets.y || '0');
//
//          }
//        } catch (error) {
//          //Catch errors and gracefully fallback to Readium implementation...
//          console.log(error);
//        }
 
      return reader.openSpineItemElementCfi(idref, processedCFI);//Not sure what initiator does, so not using it...
    },

    convertTextOffsetsToSpatialOffsets: function (parentNode, textNode, characterOffset) {
      var $iframe,
        contentDoc,
        offset = {
          x: 0,
          y: 0
        },
        range,
        USE_RANGY = true,// && (rangy !== undefined),
        offsetRect,
        clientRects,
        // foundRange = false,
        iframeDimensions = {
          width: 0,
          height: 0
        },
        nRange,
        info;

      try {
        $iframe = this.getFrames();
        iframeDimensions.width = $iframe.width();
        iframeDimensions.height = $iframe.height();
        contentDoc = $iframe[0].contentDocument;

        if (USE_RANGY) {
          range = rangy.createRange(contentDoc);
          range.setStart(textNode, characterOffset);
          range.collapse(true); //toStart = true

          nRange = (range.nativeRange || range);
          offsetRect = nRange.getClientRects()[0];
          clientRects = parentNode.getClientRects();

          info = Utility.getRectOffsetsInParent(clientRects, offsetRect, iframeDimensions);

          offset.y = Math.floor(info.offsetToRect * 100 / info.totalHeight);
        }
      } catch (error) {
        console.log(error);
      }

      return offset;
    },

    getChildNodeFromSpatialOffsets: function (parentNode, offsets) {
      var $iframe,
        contentDoc,
        offsetHeight,
        childNode,
        childInfo,
        nodeInfo,
        offset;

      $iframe = this.getFrames();
      contentDoc = $iframe[0].contentDocument;

      try {        
        //If the coordinates are not scaled to parent's system, do it...
        if (offsets.scaled) {
          offsetHeight = offsets.y;
        } else {
          offsetHeight = Utility.scaleHeightOffset(parentNode, offsets.y);
        }
        
        //Find which child this offset falls under...
        childInfo = Utility.getChildAtOffsetV(parentNode, offsetHeight, contentDoc, { height: $iframe.height(), width: $iframe.width() });
        
        //This is not in any child rect, that means the parent is directly under offset...
        if (!childInfo || !childInfo.node) {
          return {
            node: parentNode
          };
        } 
        
        //Process this child...
        childNode = childInfo.node;
        offsetHeight = childInfo.relativeOffset;

        if (childNode.nodeType !== Node.TEXT_NODE) {
          //Re-run this code for this node...
          nodeInfo = AdobeRDMHelper.getChildNodeFromSpatialOffsets(childNode, { x: 0, y: offsetHeight, scaled: true });
          return nodeInfo;
        }
        offset = Utility.getCharacterOffsetAtOffsetV(childNode, offsetHeight, contentDoc);
        return {
          node: childNode,
          offset: offset
        };
      } catch (error) {
        console.log(error);
      }
    },

    /**Returns a CFI + Node pair for each of start and end 
     * @param {Boolean} convert True makes sense only for current content doc*/
    getNodeRangeFromSpatialCFI: function (idref, partialCFIStart, partialCFIEnd, convert) {     
      var conversionInfo,
        range = {
          start: {
            node: null,
            CFI: partialCFIStart
          },
          end: {
            node: null,
            CFI: partialCFIEnd
          }
        },
        plainCFI,
        parentNode,
        expandedCfiInfo,
        contentDoc = AdobeRDMHelper.getCurrentDocument(idref);

      //Removing try...catch since we don't expect anything to fail and want to be notified...
      plainCFI = Utility.stripSpatialCFI(partialCFIStart);
      expandedCfiInfo = AdobeRDMHelper.getElementByExpandedCfi(plainCFI.CFI, contentDoc);
      if (expandedCfiInfo.reloadCFI) {
          range.start.node = expandedCfiInfo.element;
          range.start.CFI = expandedCfiInfo.CFI;
      } else {
        parentNode = expandedCfiInfo.element;
        if (convert && plainCFI.foundSpatial) {
          conversionInfo = Utility.convertSpatialCFI(parentNode, plainCFI.x, plainCFI.y);
          range.start.node = conversionInfo.node;
          range.start.CFI = conversionInfo.CFI;
        } else {
          range.start.node = parentNode;
          range.start.CFI = plainCFI.CFI;
        }
      }

      plainCFI = Utility.stripSpatialCFI(partialCFIEnd);
      expandedCfiInfo = AdobeRDMHelper.getElementByExpandedCfi(plainCFI.CFI, contentDoc);
      if (expandedCfiInfo.reloadCFI) {
          range.end.node = expandedCfiInfo.element;
          range.end.CFI = expandedCfiInfo.CFI;
      } else {
        parentNode = expandedCfiInfo.element;
        if (convert && plainCFI.foundSpatial) {
          conversionInfo = Utility.convertSpatialCFI(parentNode, plainCFI.x, plainCFI.y);
          range.end.node = conversionInfo.node;
          range.end.CFI = conversionInfo.CFI;
        } else {
          range.end.node = parentNode;
          range.end.CFI = plainCFI.CFI;
        }
      }

      return range;
    },

    isFindRangeLooping: function (iteratorStartCFI, iteratorEndCFI, reverse) {
      var isLoop = false,
        relationship;

      iteratorStartCFI = AdobeCfiIterator.normalizeCFI(iteratorStartCFI);
      iteratorEndCFI = AdobeCfiIterator.normalizeCFI(iteratorEndCFI);
      relationship = AdobeCfiIterator.cfiCompare(iteratorStartCFI, iteratorEndCFI);
      if (relationship === AdobeCfiIterator.CFIRelationships.EqualTo) {
        //Equal to is a loop in both cases...
        isLoop = true;
      } else if (relationship === AdobeCfiIterator.CFIRelationships.GreaterThan
          || relationship === AdobeCfiIterator.CFIRelationships.ChildOf) {
      
        if (!reverse) {
          isLoop = true;
        }
      } else if (reverse) {
        isLoop = true;
      }

      return isLoop;
    },

    returnAsynchronously: function (msgData){
       var retVal = {"msgType":"onFindTextCompleted", "msgData":msgData};
       AdobeRDMHelper.getReader().trigger(ReadiumSDK.Events.GENERIC_HOST_CALLBACK, retVal);
       return retVal;
    },
  
    findTextForCFIRange: function(textToFind, startIdref, partialCFIStart, endIdref, partialCFIEnd, searchMultiple, flags) {
      searchMultiple = searchMultiple || false;
      window.setTimeout(function(){
        var options,
            behavior = {},
            /**HELPER FUNCTIONS*/
            onSearchEnded,
            incrementSpineIndex,
            decrementSpineIndex,
            populateFindResultsInRange,
            onItemLoad,
            getNodeCfiPair;

        onSearchEnded = function (matchArray) {
          var returnJSONString = '';
          //Will not work on IE6/7 - need external package for this...
          if (!searchMultiple) {
            returnJSONString = JSON.stringify(matchArray[0]);
          }
          else {
            returnJSONString = JSON.stringify(matchArray);
          }
          AdobeRDMHelper.returnAsynchronously(returnJSONString);
        };

        incrementSpineIndex = function (spineItem) {
          var spineIndex = spineItem.index + 1;
          spineIndex %= spineItem.spine.items.length;
          return spineIndex;
        };
        
        decrementSpineIndex = function (spineItem) {
          var spineIndex = spineItem.index - 1;
          if (spineIndex < 0) {
            spineIndex += spineItem.spine.items.length; 
          }
          return spineIndex;
        };

        populateFindResultsInRange = function (range, rangeIdref, matchArray) {
          //Start node, start CFI, lowerBoundCFI, upperBoundCFI
          var partialMatch = [],
            iterator,
            j;

          //Start node, start CFI, lowerBoundCFI, upperBoundCFI
          if (!behavior.reverse) {
            iterator = new AdobeCfiTextIterator(range.start.node, range.start.CFI, range.start.CFI, range.end.CFI);
          } else {
            iterator = new AdobeCfiTextIterator(range.start.node, range.start.CFI, range.end.CFI, range.start.CFI);
          }

          partialMatch = behavior.partialMatchFunction(textToFind, iterator, searchMultiple);
          for (j = 0; j < partialMatch.length; ++j) {
            matchArray.push(partialMatch[j]);
            matchArray[matchArray.length - 1].idref = rangeIdref;
          }
        };

        onItemLoad = function (contentDoc) {
          try {
            /**Pass the dom through the switchers code, which modifies it on rendering (Bug #3832819)*/
            ReadiumSDK.Models.Switches.apply(contentDoc);
            
            //First, retrieve any stored state info for this spine...
            var currentState = behavior.states[behavior.spineItem.index];
            if (!currentState) {
              currentState = behavior.states[behavior.spineItem.index] = {};
            }

            //Assume the full document has to be searched and set the range...
            var range = AdobeRDMHelper.getElementRangeForDocument(contentDoc),
              looping = false,
              reachedEnd = false,
              abortSearch = false,
              nextSpineIndex;

            //Reverse range if required to keep rest of algo consistent...
            if (behavior.reverse) {
              AdobeRDMHelper.reverseRange(range);
            }


            reachedEnd = (behavior.endSpineItem.index === behavior.spineItem.index);

            /*If neither start not end exist, use the full range. [Intermediate Spine Items]
             *Ie we have any one of start or end, use that. [TOC based search list]
             *If we have both, and range forms a loop set range according to first or last traversal. [Most common case]
             *If we have both, and range does not form a loop, use this range. [Trivial case]
             *REFACTOR: If start exists, use it. If end exists it, use. Simpler logic.
             */
            if (currentState.startCFI && !currentState.endCFI) {
              range.start.CFI = currentState.startCFI;
              range.start.node = AdobeRDMHelper.getElementByPartialCfi(range.start.CFI, contentDoc);
            }
            else if (!currentState.startCFI && currentState.endCFI) {
              range.end.CFI = currentState.endCFI;
              range.end.node = AdobeRDMHelper.getElementByPartialCfi(range.end.CFI, contentDoc);
            }
            else if (currentState.startCFI && currentState.endCFI) {
              looping = AdobeRDMHelper.isFindRangeLooping(currentState.startCFI, currentState.endCFI, behavior.reverse);
              if (looping) {
                if (currentState.seen) {
                  range.end.CFI = currentState.endCFI;
                  range.end.node = AdobeRDMHelper.getElementByPartialCfi(range.end.CFI, contentDoc);
                } else {
                  //Check for the special case of first spine item when looping...
                  reachedEnd = false;

                  range.start.CFI = currentState.startCFI;
                  range.start.node = AdobeRDMHelper.getElementByPartialCfi(range.start.CFI, contentDoc);
                }
              } else {
                range.start.CFI = currentState.startCFI;
                range.start.node = AdobeRDMHelper.getElementByPartialCfi(range.start.CFI, contentDoc);

                range.end.CFI = currentState.endCFI;
                range.end.node = AdobeRDMHelper.getElementByPartialCfi(range.end.CFI, contentDoc);
              }
            }

            if (reachedEnd && !currentState.endCFI) {
            /**Since find's end CFI is exclusive, for the last spine item, the default document end should be 
             * the same as start (i.e, no search occurs). Thus, Find [idref1!"", idref3!"")
             * searches through idref1 and idref2 but not idref3*/
              abortSearch = true;
            }
              
            if (!abortSearch) {
              populateFindResultsInRange(range, behavior.spineItem.idref, behavior.fullMatch);
              currentState.seen = true;
              currentState.seenRange = {start: range.start.CFI, end: range.end.CFI};
            }
            
            if (reachedEnd || (behavior.fullMatch.length && !searchMultiple)) {
              onSearchEnded(behavior.fullMatch);
            } else {
              //If searchMultiple is true or fullMatch is empty, and more spines left then continue...
              nextSpineIndex = behavior.nextSpineFunction(behavior.spineItem);
              behavior.spineItem = behavior.spineItem.spine.item(nextSpineIndex);
              AdobeRDMHelper.getSpineData(behavior.spineItem, onItemLoad);
            }
          } catch (error) {
            console.error(error);
            AdobeRDMHelper.returnAsynchronously('');
          }
        };

        getNodeCfiPair = function (contentDoc, cfi, convertSpatial) {
          var conversionInfo,
            nodepair = {
                node: null,
                CFI: partialCFIStart
            },
            cfiInfo,
            expandedCfiInfo;

          cfiInfo = Utility.stripSpatialCFI(cfi);
          expandedCfiInfo = AdobeRDMHelper.getElementByExpandedCfi(cfiInfo.CFI, contentDoc);

          //In an ideal scenario with standard, non-spatial CFI, we are done with this...
          nodepair.node = expandedCfiInfo.element;
          nodepair.CFI = cfiInfo.CFI;
          
          if (expandedCfiInfo.reloadCFI) {
            //The cfi we had was non-standard. Replace with the standard CFI returned...
            nodepair.CFI = expandedCfiInfo.CFI;
          } else {
            /**If a spatial CFI was passed, we have currently stripped that and got the parent...
             * So, convert the spatial CFI and reload nodepair*/
            if (convertSpatial && cfiInfo.foundSpatial) {
              conversionInfo = Utility.convertSpatialCFI(nodepair.node, cfiInfo.x, cfiInfo.y);
              nodepair.node = conversionInfo.node;
              nodepair.CFI = conversionInfo.CFI;
            }
          }

          return nodepair;
        };

        try {
        //###
          if (!textToFind || textToFind === ' ') {
            throw new Error('AdobeRDMHelper: No search string found');
          }           
          options = SearchHelper.decodeFlags(flags);

          //We assume options.shouldWrap is set by default
          behavior.reverse = options.reverse;
          if (behavior.reverse) {
            behavior.nextSpineFunction = decrementSpineIndex;
            behavior.partialMatchFunction = AdobeRDMHelper.findInReverseRange;
          } else {
            behavior.nextSpineFunction = incrementSpineIndex;
            behavior.partialMatchFunction = AdobeRDMHelper.findInRange;
          }

          behavior.spineItem = this.getSpineItemFromIdref(startIdref);
          behavior.endSpineItem = this.getSpineItemFromIdref(endIdref);
          if( !behavior.spineItem || !behavior.endSpineItem ) {
            throw new Error('AdobeRDMHelper: Invalid spine item reference');
          }

          //shakrish
          var currentSpineItem,
            nodeCfiPair;

          //If start or end is in the current spine item, we may have to deal with spatial CFIs...
          /*From: reader_view > refreshAnnotations 
           *An alternate way: see reader_view > openPageNext > lastOpenPage.idref 
           */
          currentSpineItem = this.getReader().getLoadedSpineItems()[0];

          //A map of states corresponsing to each spine index...
          behavior.states = {};
          
          behavior.states[behavior.spineItem.index] = {};
          if (partialCFIStart && partialCFIStart.length > 0) {
            if (startIdref === currentSpineItem.idref) {
              nodeCfiPair = getNodeCfiPair(this.getCurrentDocument(), partialCFIStart, true);
              behavior.states[behavior.spineItem.index].startCFI = nodeCfiPair.CFI;
            } else {
              behavior.states[behavior.spineItem.index].startCFI = partialCFIStart;
            }
          }

          //Take care not to recreate when startidref === endidref
          behavior.states[behavior.endSpineItem.index] = behavior.states[behavior.endSpineItem.index] || {};
          if (partialCFIEnd && partialCFIEnd.length > 0) {
            if (endIdref === currentSpineItem.idref) {
              nodeCfiPair = getNodeCfiPair(this.getCurrentDocument(), partialCFIEnd, true);
              behavior.states[behavior.endSpineItem.index].endCFI = nodeCfiPair.CFI;
            } else {
              behavior.states[behavior.endSpineItem.index].endCFI = partialCFIEnd;
            }
          }
          
          behavior.fullMatch = [];

          this.getSpineData(behavior.spineItem, onItemLoad);
        } catch (error) {
          console.error(error);
          return AdobeRDMHelper.returnAsynchronously('');
        }

      }.bind(this), 0);
      return '';
    },

    splitRangeCFI: function(rangeCFI)
    {
        var range = {
          start: {
            CFI: '',
            node: null,
            offset: 0
          },
          end: {
            CFI: '',
            node: null,
            offset: 0
          }
        },
        splitCfi = rangeCFI.split(',');
        
      range.start.CFI = splitCfi[0] + splitCfi[1];
      range.end.CFI = splitCfi[0] + splitCfi[2];
      return range;
    },

    getElementRangeForStartAndEndCFI: function(idref, startCFI, endCFI)
    {
      var spineItem = this.getSpineItemFromIdref(idref),
        range = {
          start: {
            CFI: startCFI,
            node: null,
            offset: 0
          },
          end: {
            CFI: endCFI,
            node: null,
            offset: 0
          }
        },
        tracer = new Utility.Tracer('getElementRangeForStartAndEndCFI');

      try {
        if (spineItem) {
          range.start.node = this.getElementByCfi(spineItem, range.start.CFI);
          range.end.node = this.getElementByCfi(spineItem, range.end.CFI);
          tracer.log('Got range nodes');

          if (range.start.node.nodeType === Node.TEXT_NODE) {
            range.start.offset = Utility.getCharacterOffsetFromCFI(range.start.CFI);
          }
          if (range.end.node.nodeType === Node.TEXT_NODE) {
            range.end.offset = Utility.getCharacterOffsetFromCFI(range.end.CFI);
          }
        }
      } catch (error) {
       tracer.log(error.message);
       throw new Error (tracer.trace());
      }

      return range;
    },

    getElementRangeForDocument: function (contentDoc) {
      var bodyElem = contentDoc.getElementsByTagName('body')[0],
        firstNode,
        lastNode,
        firstCFI,
        lastCFI;
      
      firstNode = bodyElem.firstChild;
      if (firstNode.nodeType === Node.TEXT_NODE) {
        firstCFI = AdobeRDMHelper.generateTextElementCFI(firstNode, 0);
      } else {
        firstCFI = AdobeRDMHelper.generateElementCFI(firstNode);
      }

      lastNode = bodyElem.lastChild;
      while (lastNode.lastChild !== null) { // Go to last child recursively till you reach the element which has lastChild = null
        lastNode = lastNode.lastChild;
      } if (lastNode.nodeType === Node.TEXT_NODE) {
        /*NOTE shakrish This will have the blacklisted elements bug if used on non-pristine DOM*/
        lastCFI = AdobeRDMHelper.generateTextElementCFI(lastNode, lastNode.wholeText.length);
      } else {
        lastCFI = AdobeRDMHelper.generateElementCFI(lastNode);
      }

      return {
        start: {
          node: firstNode,
          CFI: firstCFI
        },
        end: {
          node: lastNode,
          CFI: lastCFI
        }
      };
    },

    getCurrentDocumentEndCfi: function () {
      var currentDoc,
        currentDocumentFullRange,
        cfi;

      currentDoc = this.getCurrentDocument();
      currentDocumentFullRange = this.getElementRangeForDocument(currentDoc);

      cfi = currentDocumentFullRange.end.CFI || "";
      return cfi; 
    },

    reverseRange: function (range) {
      var temp = range.end;
      range.end = range.start;
      range.start = temp;
    },

    /**Convenience method to get node range from Range CFI*/
    getRangeForCFI: function (idref, rangeCFI) {
      var result,
        tracer = new Utility.Tracer('getRangeForCFI');
      try {
        var range = this.splitRangeCFI(rangeCFI);
        tracer.log('Split Range CFI');

        result = this.getElementRangeForStartAndEndCFI(idref, range.start.CFI, range.end.CFI);
      } catch (error) {
        tracer.log(error.message);
        throw new Error (tracer.trace());
      }
      return result;
    },

    mouseDownEventListener : null,

    resetSelection: function () {

      var reader = this.getReader();
      reader.resetSelection();

      var contentDoc = this.getCurrentDocument();
      var $body = $("body", contentDoc);

      if(this.mouseDownEventListener) {
        $body[0].removeEventListener('mousedown',this.mouseDownEventListener);
        $body[0].removeEventListener('touchstart',this.mouseDownEventListener);

        this.mouseDownEventListener = null;
      }
      return '';
    },


    setSelection: function (spineIdRef, rangeCFI) {
      var reader = this.getReader();
      reader.setSelection(rangeCFI, spineIdRef);

      var contentDoc = this.getCurrentDocument();
      var $body = $("body", contentDoc);

      this.mouseDownEventListener = function() { this.resetSelection();}.bind(this);

      $body[0].addEventListener('mousedown',this.mouseDownEventListener);
      $body[0].addEventListener('touchstart',this.mouseDownEventListener);

      return JSON.stringify({
        CFI: rangeCFI,
        id: 0,
        idref: spineIdRef 
      });
    },

    addHighlight: function (spineIdRef, Cfi, id, type, styles) {
      var result = this.getReader().addHighlight(spineIdRef, Cfi, id, type, styles);
      if(result)
      {
        return JSON.stringify({
          CFI: result.CFI,
          id: result.id,
          idref: result.idref 
        });
      }
      return undefined;
    },

    /**@param [{ idref, rangeCFI }] annotationArray*/
    updateAnnotationHighlights: function (annotationArray) {
      var msg = "Got " + annotationArray.length + " highlights";

      var i, eachAnnotation, resultArray, result;

      resultArray = [];
      for (i = 0; i < annotationArray.length; ++i) {
        eachAnnotation = annotationArray[i];
        result = this.getReader().addHighlight(
          eachAnnotation.idref,
          eachAnnotation.rangeCFI,
          eachAnnotation.highlightId,
          "highlight"
        );
        resultArray.push(result);
      }

      return JSON.stringify(resultArray);
    },


    // When the text offset is interfered by some non-text nodes(this will happen when there is a highlight created),
    // Readium does not return the actual text node which has the text associated with the offset.
    // This method will go through all the sibling nodes and returns the actual text node and the new offset.
    getActualTextNodeAndOffset: function(textNode, textOffset)
    {
      var result = {
        node: textNode,
        offset: textOffset
      };

      if(textNode.nodeType === Node.TEXT_NODE)
      {
        while (textNode && textOffset > textNode.nodeValue.length) {
          textOffset -= textNode.nodeValue.length;
          /**Find the next text node... 
           * Ignore any non-text nodes. Since we assume CFI is valid they're probably 'blacklisted' ones...
           */
           do {
            textNode = textNode.nextSibling;
           } while (textNode && textNode.nodeType !== Node.TEXT_NODE);
        }
        result.node = textNode;
        result.offset = textOffset;
      }
      return result;
    },

    getNearestTextNodeAndOffset: function(node, offset, forwardDirection)
    {
      var result = {
        node: node,
        offset: offset
      }, tmp=node;

      if(node.nodeType === Node.TEXT_NODE) {
        return result;
      }

      /*global NodeFilter*/
      var iter = AdobeRDMHelper.createTreeWalker(document, document, NodeFilter.SHOW_TEXT, {acceptNode: function (node) {
        //only accept nodes that have content
        // other than whitespace
        if ( ! /^\s*$/.test(node.data) ) {
          return NodeFilter.FILTER_ACCEPT;
        }
      }}, false);
      iter.currentNode = tmp;
      while(tmp && tmp.nodeType !== Node.TEXT_NODE)
      {
        if(forwardDirection) {
          tmp = iter.nextNode();
        }
        else {
          tmp = iter.previousNode();
        }
      }

      if(tmp && tmp.nodeType === Node.TEXT_NODE)
      {
        result.node = tmp;
        if(forwardDirection) {
          result.offset = 0;
        }
        else {
          result.offset = (tmp.nodeValue.length>0)?tmp.nodeValue.length-1:0;
        }
      }
      return result;
    },

    /*
     * Given a node in the DOM, this method will return the idref of the spine item in which the node is present.
     */
    getSpineItemIdrefForNode: function(node)
    {
      /*jslint unparam:true*/
      var retVal = "";
      try{
        var v = node.ownerDocument.baseURI.split("/");
        var baseURIName = v[v.length-1];
        var loadedSpineItems = AdobeRDMHelper.getReader().getLoadedSpineItems();
        var cnt = loadedSpineItems.length;
        var i;
        var href;
        for (i = 0; i < cnt; i++) 
        {
          href = loadedSpineItems[i].href;
          if(href.lastIndexOf(baseURIName) ===  href.length-baseURIName.length)
          {
            retVal = loadedSpineItems[i].idref;
          }
        }
      }
      catch(error)
      {
        console.error("Error in getSpineItemIdrefForNode(), node: " + node + " : " + error);
      }
      return retVal;
    },

    /*
     * Given a spie item idref and a text node CFI with offset, this method returns the idref and CFI of the word boundaries of it
     */
    getWordBoundaryCFIRange: function(idref, partialCFI)
    {
      var spineItem = AdobeRDMHelper.getSpineItemFromIdref(idref),
          node, offset=0, wordBoundary,
          range = {
            idref: idref,
            startCFI: partialCFI,
            endCFI: partialCFI
          };

      try{
        if (spineItem) {
          node = AdobeRDMHelper.getElementByCfi(spineItem, partialCFI);

          if (node.nodeType === Node.TEXT_NODE) {
            offset = Utility.getCharacterOffsetFromCFI(partialCFI);
          }      
        }
        wordBoundary = AdobeRDMHelper.getWordBoundary(node, offset, true);
        range = {
              idref: idref,
              startCFI: wordBoundary.wordStart.contentCFI,
              endCFI: wordBoundary.wordEnd.contentCFI
            };
      }catch(error)
      {
        console.error("Error in getWordBoundaryCFIRange(" + idref + ", " + partialCFI + "): " + error);
      }
    //console.error("getWordBoundaryCFIRange() partialCFI: "+partialCFI + " retVal: " + JSON.stringify(range));//temp
      return JSON.stringify(range);
    },

    /*
     * Given a text node and offset, this method returns the {wordStart:{node:<x>, offset:<x>, idref:"", contentCFI:""}, wordEnd:{node:<x>, offset:<x>, idref:"", contentCFI:""}}
     */
    getWordBoundary: function(node, offset, isCFINeeded)
    {
      var r = AdobeRDMHelper.getActualTextNodeAndOffset(node, offset),
          retVal = { wordStart: {node:r.node, offset:r.offset, idref:"", contentCFI:""}, wordEnd:{node:r.node, offset:r.offset, idref:"", contentCFI:""}},
          tmp=r.node,
          iter = AdobeRDMHelper.createTreeWalker(document, document, NodeFilter.SHOW_TEXT, {
            acceptNode: function (node){
              /*jslint unparam:true*/
              //only accept nodes that have content
              // other than whitespace
                return NodeFilter.FILTER_ACCEPT;
            }
          }, false);

          try{
            iter.currentNode = r.node;
            if(r.offset < 0) {
              r.offset = 0;
            }
            if(r.offset >= tmp.data.length) {
              r.offset = tmp.data.length-1;
            }

            // Find the space to the left of the given offset
            var wordStartOffset = r.offset;
            // If the first character itself is a space go to a non-space character and start finding the word boundary
            while(/\s/.test(tmp.data[wordStartOffset]))
            {
              if(++wordStartOffset >= tmp.data.length)
              {
                tmp = iter.nextNode();
                wordStartOffset = 0;
              }            
            }
            r.offset = wordStartOffset;
            r.node = tmp;
            iter.currentNode = r.node;
            var prevNode = tmp;
            var prevOffset = wordStartOffset;            
            while(true)
            {
              // If the char is a white space, we got the boundary
              if(/\s/.test(tmp.data[wordStartOffset])) {
                break;
              }

              prevOffset = wordStartOffset;
              prevNode = tmp;              
              if(--wordStartOffset < 0)
              {
                do {
                  tmp = iter.previousNode();
                } while(tmp && (tmp.data.length === 0));
                if(!tmp) {
                  break;
                }
                wordStartOffset = tmp.data.length-1;
              }
            }
            retVal.wordStart.node = prevNode;
            retVal.wordStart.offset = prevOffset;

            // Find the space to the right of the given offset
            tmp = r.node;
            iter.currentNode = r.node;
            var wordEndOffset = r.offset;
            prevNode = tmp;
            prevOffset = wordEndOffset;            
            while(true)
            {
              if(/\s/.test(tmp.data[wordEndOffset])) {
                break;
              }
              prevOffset = wordEndOffset;
              prevNode = tmp;              
              if(++wordEndOffset >= tmp.data.length) {
                tmp = iter.nextNode();
                if(!tmp) {
                  break;                
                }
                wordEndOffset = 0;
              }
            }
            retVal.wordEnd.node = prevNode;
            retVal.wordEnd.offset = prevOffset; 

            if(isCFINeeded)
            {
              var idref = AdobeRDMHelper.getSpineItemIdrefForNode(retVal.wordStart.node);
              var contentCFI = AdobeRDMHelper.generateTextElementCFI(retVal.wordStart.node, retVal.wordStart.offset);
              retVal.wordStart.idref = idref;
              retVal.wordStart.contentCFI = contentCFI;
              if(retVal.wordStart.node !== retVal.wordEnd.node) {
                idref = AdobeRDMHelper.getSpineItemIdrefForNode(retVal.wordEnd.node);
              }
//              else{
//                contentCFI = contentCFI.split(":")[0] + ":" + retVal.wordEnd.offset;
//              }
              contentCFI = AdobeRDMHelper.generateTextElementCFI(retVal.wordEnd.node, retVal.wordEnd.offset);
              retVal.wordEnd.idref = idref;
              retVal.wordEnd.contentCFI = contentCFI;
            }           
          }catch(error)
          {
            console.error("Error in getWordBoundary node: " + node + " offset: " + offset + " : " + error);
          }
          return retVal;
    },

    getBookMargin: function(){
          var retVal = {left:0, right:0, top:0, bottom:0};
          var settings = this.getReader().viewerSettings();
          var isReflowLayout = !this.getReader().isCurrentViewFixedLayout();
          if(isReflowLayout){
            retVal.left = settings.getMarginLeftCSSPx();
            retVal.top = settings.getMarginTopCSSPx();
            retVal.right = settings.getMarginRightCSSPx();
            retVal.bottom = settings.getMarginBottomCSSPx();
          }
          else
          {
            var fixedFrame = $("#fixed-book-frame")[0];
            if(fixedFrame){
              retVal.left = parseInt(fixedFrame.style.left, 10);
              retVal.top = parseInt(fixedFrame.style.top, 10);               
            }
          }
          return retVal;
    },

    getWebViewDimensions: function(){
          var retVal = {};
          var settings = this.getReader().viewerSettings();
          retVal.width = settings.viewPortWidth || window.screen.width;
          retVal.height = settings.viewPortHeight || window.screen.height;
          return retVal;
    },

    /*
     * Given a point (x, y) in viewport space, this method will return the CFI and idref of the node 
     * under that point. If its a text node, it returns the exact text offset also. 
     * Returns the result in the same format as what is returned by ReadiumSDK.reader.bookmarkCurrentPage()
     */
    getBookmarkFromPoint: function(x, y, dontScaleForHTMLWindowDims, dontStringifyResult){
      var retVal = {idref:'', contentCFI:''},
          pos = null, r, v, baseURIName, loadedSpineItems, cnt, i, href;
      try{
        if(!dontScaleForHTMLWindowDims)
        {
          var margin = AdobeRDMHelper.getBookMargin();
          var webViewDims = AdobeRDMHelper.getWebViewDimensions();
         if(ReadiumSDK._Android && window.devicePixelRatio){
			//On Android, webview dimensions are already scaled. Dont scale them again
            webViewDims.width /= window.devicePixelRatio;
            webViewDims.height /= window.devicePixelRatio;
         }
          // TODO Added for iOS. Need to verify if using window.screen.width hold good for win/mac etc when the webview is not fullscreen
          var scaleX = window.innerWidth/webViewDims.width;
          var scaleY = window.innerHeight/webViewDims.height;
          // Apply the scaling to handle the retina displays and other higher resolution displays
          if(window.devicePixelRatio){
            scaleX /= window.devicePixelRatio;
            scaleY /= window.devicePixelRatio;
          }
          x *= scaleX;
          y *= scaleY;
          x = x-margin.left;
          y = y-margin.top;
        }
        loadedSpineItems = this.getReader().getLoadedSpineItems();        
        //TODO we need to handle cases where multiple spine items are shown on the single viewport
        var highlightNodeList = [],
          isHighlightNode;
        while(true)
        {
          cnt = loadedSpineItems.length;
          for(i=0;i<cnt;i++)
          {
            //rangy.positionFromPoint() sometimes throws. So lets keep it in try{}catch{} block
            try{
              pos = rangy.positionFromPoint(x, y, this.getCurrentDocument(loadedSpineItems[i].idref));
              if(pos.node) {
                break; // We found the position object. just break
              }
            } catch(ignore) {
            }
          }
          if(pos.node.nodeType === Node.TEXT_NODE)
          {
            r = AdobeRDMHelper.getActualTextNodeAndOffset(pos.node, pos.offset);
            retVal.contentCFI = AdobeRDMHelper.generateTextElementCFI(r.node, r.offset);
            //console.error("getBookmarkFromPoint x:" + x + ", y: " + y + ", CFI: " + retVal.contentCFI);//temp
            //console.error("start data: " + r.node.data.substr(r.offset, 15) + "------ end data: " + r.node.data.substr((r.offset>15)?r.offset-15:0, 15));//temp

            break;
          }

          isHighlightNode = pos.node.classList.contains("highlight") || pos.node.classList.contains("hover-highlight") || pos.node.classList.contains("selection-highlight");
          if(isHighlightNode)
          {
            // We are under a highlight rect. Hide it and try again to get the actual element.
            pos.node.style.visibility = "hidden";
            highlightNodeList.push(pos.node);
          }
          else{
            retVal.contentCFI = AdobeRDMHelper.generateElementCFI(pos.node);
            //console.error("getBookmarkFromPoint x:" + x + ", y: " + y + ", found non text node, CFI: " + retVal.contentCFI);//temp
            break;
          }
        }

        // Restore the visibility state of highlight
        for(i=0;i<highlightNodeList.length;i++)
        {
          highlightNodeList[i].style.visibility = "visible";
        }

        v = pos.node.ownerDocument.baseURI.split("/");
        baseURIName = v[v.length-1];
        cnt = loadedSpineItems.length;
        for (i=0;i<cnt;i++) {
          href = loadedSpineItems[i].href;
          if( href.lastIndexOf(baseURIName) ===  href.length-baseURIName.length)
          {
            retVal.idref = loadedSpineItems[i].idref;
          }
        }
      }catch(error)
      {
        console.log("Error in getCFIForPosition(" + x + ", " + y + "): " + error);
      }
      if(dontStringifyResult) {
        return retVal;
      }
      return JSON.stringify(retVal);
    },    

    getBoxesForCFIRange: function(idref, partialCFIStart, partialCFIEnd, dontScaleForNativeWinDims) {
      return JSON.stringify(this.getBoxesDataForCFIRange(idref, partialCFIStart, partialCFIEnd, dontScaleForNativeWinDims));
    },


    /**NOTE
     * Returns the rects of the CFI range. The CFI range should refer to text nodes only*/
    getBoxesDataForCFIRange: function(idref, partialCFIStart, partialCFIEnd, dontScaleForNativeWinDims) {
      /*jslint vars: true white: true*/
      var retVal = { count:0, boxes:[]},
        range, nRange, boxes, i, scaleX, scaleY,
        USE_RANGY = true;// && (rangy !== undefined);
      // var box = {left:0, top:0, right:0, bottom:0};  
      var margin = {left:0, top:0, right:0, bottom:0};  
      if (USE_RANGY)
      {
        var txtNodeRange = AdobeRDMHelper.getNodeRangeFromSpatialCFI(idref, partialCFIStart, partialCFIEnd, true);
        var cfiRange = AdobeRDMHelper.getElementRangeForStartAndEndCFI(idref, txtNodeRange.start.CFI, txtNodeRange.end.CFI);

        try {
          var start = AdobeRDMHelper.getActualTextNodeAndOffset(cfiRange.start.node, cfiRange.start.offset);
          var end = AdobeRDMHelper.getActualTextNodeAndOffset(cfiRange.end.node, cfiRange.end.offset);          
          if (start.node && end.node) {
            range = rangy.createRange(this.getCurrentDocument(idref));
            
            range.setStart(start.node, start.offset);
            range.setEnd(end.node, end.offset);
            nRange = (range.nativeRange || range);
            boxes = nRange.getClientRects();

            if(!dontScaleForNativeWinDims)
            {
              var webViewDims = AdobeRDMHelper.getWebViewDimensions();
              // TODO Added for iOS. Need to verify if using window.screen.width hold good for win/mac etc when the webview is not fullscreen
              scaleX = webViewDims.width/window.innerWidth;
              scaleY = webViewDims.height/window.innerHeight;

              margin = AdobeRDMHelper.getBookMargin();

              // Apply the scaling to handle the retina displays and other higher resolution displays
              if(window.devicePixelRatio){
                scaleX *= window.devicePixelRatio;
                scaleY *= window.devicePixelRatio;
              }              

            } else {
              scaleX = scaleY = 1;
            }            
            retVal.count = boxes.length;

            
            for(i=0;i<retVal.count;i++)
            {                                        
              retVal.boxes.push({left:(boxes[i].left+margin.left)*scaleX, top:(boxes[i].top+margin.top)*scaleY, right:(boxes[i].right+ margin.left)*scaleX, bottom:(boxes[i].bottom+ margin.top)*scaleY});                                                      
            }
          } 
        }catch (error) {
          console.log("Error in getTextForCFIRange "+error);
        }
         //console.error("getBoxes partialCFIStart: " + partialCFIStart + ", partialCFIEnd: " + partialCFIEnd);//temp
         //console.error("getBoxes retVal: " + JSON.stringify(retVal));//temp
         //console.error("getBookmarkFromPoint1: [left,top]="+JSON.stringify([retVal.boxes[0].left, retVal.boxes[0].top])+ " \n" + JSON.stringify(AdobeRDMHelper.getBookmarkFromPoint(retVal.boxes[0].left, retVal.boxes[0].top)));//temp
        return retVal;
      }
    },    

    /**NOTE
     * Currently runs only for active spine item's document*/
    getTextForCFIRange: function(idref, partialCFIStart, partialCFIEnd) {
      /*jslint vars: true white: true*/
      var rangeData = '',
        range,
        USE_RANGY = true;// && (rangy !== undefined);
      if (USE_RANGY)
      {
        var txtNodeRange = AdobeRDMHelper.getNodeRangeFromSpatialCFI(idref, partialCFIStart, partialCFIEnd, true);
        var cfiRange = AdobeRDMHelper.getElementRangeForStartAndEndCFI(idref, txtNodeRange.start.CFI, txtNodeRange.end.CFI);

        try {
          var start = AdobeRDMHelper.getActualTextNodeAndOffset(cfiRange.start.node, cfiRange.start.offset);
          var end = AdobeRDMHelper.getActualTextNodeAndOffset(cfiRange.end.node, cfiRange.end.offset);          
          if (start.node && end.node) {
            range = rangy.createRange(this.getCurrentDocument(idref));
            
            range.setStart(start.node, start.offset);
            range.setEnd(end.node, end.offset);
            rangeData = range.toString();
          } 
        }catch (error) {
          console.log("Error in getTextForCFIRange "+error);
        }
        return rangeData;
      }
      
      var spineItem = this.getSpineItemFromIdref(idref);
      
      if( !spineItem ) {
        return undefined;
      }

      try {
        var iterators,
          i,
          appendTextFn = function (node, CFI, text) {
            /*jslint unparam: true*/
            rangeData += text;
          };

        range = AdobeRDMHelper.getNodeRangeFromSpatialCFI(idref, partialCFIStart, partialCFIEnd, true);
        iterators = AdobeRDMHelper.getTextIteratorsFromCFIRange(range.start.CFI, range.end.CFI, range.start.node);
        for (i = 0; i < iterators.length; ++i) {
          iterators[i].traverse(appendTextFn);
        }
      } catch (error) {
        console.log(error);
        rangeData = '';
      }
      return rangeData;
    },

    getScreenEndCFI: function() {
      return this.getReader().getScreenEndCFI();
      // var $iframe = $("#epubContentIframe");
      // if( $iframe.length <= 0 )
      //   console.error("Could not find epubContentIframe");

      // var rootElement = $iframe[0].contentDocument.documentElement;
      // var $elements = $("body", rootElement).find(":not(iframe)").contents();
      // if( $elements.length > 0 )
      // {
      //   var element = $elements[$elements.length-1];
      //   var cfi;
      //   if( element.nodeType === Node.TEXT_NODE)
      //     cfi = EPUBcfi.Generator.generateCharacterOffsetCFIComponent(element, element.data.length);
      //   else
      //     cfi = EPUBcfi.Generator.generateElementCFIComponent(element);

      //   console.log(cfi);
      //   return cfi;
      // }
      // return undefined;
    },

    navigateToNthScreenFromCurrent_Info: null,
 
     navigateToNthScreenFromCurrent: function(n){
      try{
        this.getReader().navigateToNthScreenFromCurrent(n);
      }catch(error){ this.navigateToNthScreenFromCurrent_Info = null; }
     },

    /**@public Get the data of the given spine item as a DOM Tree
     * @param {SpineItem} item */
    getSpineData: function (item, onLoad) {
      /*jslint unparam: true*/
      var spine = this.getReader().spine(),
        url = spine.getItemUrl(item);

      jQuery.ajax({
        url: url,
        dataType: 'xml',
        success: onLoad,
        timeout: 10000,
        error: function(jqXHR, textStatus, errorThrown) {
          /*jslint unparam: true*/
          console.log('AdobeRDMHelper: Get Spine Data timed out');
        },
        async: true
      });
    },

    /**NOTES
     * TODO: Now that we deal with whitespaces in search string differently, the prefix handling has to change
     * TODO: Handle the CFI when multiple prefix-nodes are traversed
     */
    findInRange: function (textToFind, iterator, searchMultiple) {
      /**
       * START PREFIX MANAGER SECTION
       * TODO: Refactor this into individual, file-scope class
       */
      /**@public
       * Utility class to manage CFI, prefixes and context for search results*/
      var PrefixManager = function (minLength, sentenceLimit, wordLimit) {
        this.prefixLength = minLength;

        this.sentenceLimit = (sentenceLimit && sentenceLimit > 0) ? sentenceLimit : 1;
        this.wordLimit = (wordLimit && wordLimit < 250) ? wordLimit : 250;

        this.sentenceEntries = [];
      };


      /**@public*/
      PrefixManager.sentenceEndRegex = function () {
        /* [.?!] are only possible sentence delimiters.
         * ['"] may indicate the sentence ended as a quote. Matched either 0 or 1 time.
         * \s+ is at least one whitespace.
         * TODO: Should an alternative for \s+ be $ (end of input)?
         */
        return ( /[\.\?\!]['"]{0,1}\s+/ig );
      };

      /**@public*/
      PrefixManager.wordCountRegex = function () {
        /**
         * \s* may be preceeded by whitespaces
         * \S{3,} at least 3 non-whitespaces make a word
         * \s+ at least one whitespace ends it
         */
        return ( /\s*\S{3,}\s+/ig );
      };

      /**@public
       * Represents zero or one sentence, within a single CFI*/
      PrefixManager.SentenceEntry = function (content, cfi, offset) {
        content = content || '';
        
        this.buffer = '';
        this.nodes = [];
        this.words = 0;
        this.isComplete = false;

        this.add(content, cfi, offset);
      };

      /**@public*/
      PrefixManager.SentenceEntry.prototype.add = function (content, cfi, offset) {
        var regexResult = content.match(PrefixManager.wordCountRegex());
        this.words += regexResult ? regexResult.length : 0;
        this.endsAtNonWhiteSpace = content.length && (content[content.length - 1].search(/\S/) >= 0);

        var lastNode,
          usePreviousNode = false;
        if (this.nodes.length > 0) {
          lastNode = this.nodes[this.nodes.length - 1];
          if (lastNode && lastNode.cfi === cfi && (lastNode.offset + lastNode.length) === offset) {
            usePreviousNode = true;
          }
        }
        if (usePreviousNode) {
          lastNode.length += content.length;  
        } else {
          this.nodes.push({ length: content.length, cfi: cfi, offset: offset });
        }
        
        this.buffer += content;
        return this;
      };

      /**@public*/
      PrefixManager.SentenceEntry.prototype.end = function () {
        if (this.endsAtNonWhiteSpace) {
          ++this.words;
          this.endsAtNonWhiteSpace = false;
        }
        this.isComplete = true;

        return this;
      };

      /**@public*/
      PrefixManager.SentenceEntry.prototype.getLength = function () {
        return this.buffer.length;
      };

      /**@public*/
      PrefixManager.SentenceEntry.prototype.getEntryForOffset = function (offset) {
        var offsetFromStart = 0,
          result;
        
        this.nodes.some(function (value, index, map) {
          if (index >= map.length - 1
            || (offsetFromStart <= offset && offsetFromStart + value.length > offset)) {
            result = {
              cfi: value.cfi,
              offset: offset - offsetFromStart + value.offset
            };
            return true;
          }

          offsetFromStart += value.length;
          return false;
        });

        return result;
      };
      /**PrefixManager.SentenceEntry*/

      /**@private*/
      PrefixManager.prototype.trim = function () {
        if (this.sentenceLimit >= this.sentenceEntries.length) {
          return;
        }
        
        this.sentenceEntries.splice(0, this.sentenceEntries.length - this.sentenceLimit);
        return;
      };

      /**@private*/
      PrefixManager.prototype.getLastEntry = function () {
        return this.sentenceEntries[this.sentenceEntries.length - 1];
      };

      /**@private*/
      PrefixManager.prototype.shouldStartSentence = function () {
        if (!this.sentenceEntries.length) {
          return true;
        }
        
        return this.getLastEntry().isComplete;
      };

      /**@public*/
      PrefixManager.prototype.minPrefix = function () {
        var buffer = this.getBuffer();
        return (buffer.length > this.prefixLength) ? buffer.substr(-this.prefixLength) : buffer;
      };

      /**@public*/
      PrefixManager.prototype.empty = function () {
         this.sentenceEntries = [];
      };

      /**@public*/
      PrefixManager.prototype.getBuffer = function () {
        var buffer = '';
        this.sentenceEntries.forEach(function (entry) {
          buffer += entry.buffer;
        });
        return buffer;
      };

      /**@public Add some string content, at a specific text-CFI and offset*/
      PrefixManager.prototype.add = function (content, contentCFI, offset) {
        var regex,
          newEntry,
          endIndex,
          eachSentence,
          regexResult,
          sentenceOffset;

        //Append to old incomplete sentence, or start a new one...
        if (this.shouldStartSentence()) {
          newEntry = new PrefixManager.SentenceEntry('', contentCFI, offset);
        } else {
          newEntry = this.sentenceEntries.pop();
        }

        /**Break the content up into multiple sentences (entries)
         * IMPORTANT: For explanation of RegExp.exec vs String.match, see MDN
         */
        regex = PrefixManager.sentenceEndRegex();
        regexResult = regex.exec(content);
        sentenceOffset = offset;
        while (regexResult) {
          endIndex = regexResult.index + regexResult[0].length;
          eachSentence = content.substr(0, endIndex);

          //Add this entry to the set...
          newEntry.add(eachSentence, contentCFI, sentenceOffset).end();
          sentenceOffset += eachSentence.length;
          this.sentenceEntries.push(newEntry);
          
          //Move to the next Entry...        
          content = content.substr(endIndex);
          regexResult = regex.exec(content);
          newEntry = new PrefixManager.SentenceEntry('', contentCFI, sentenceOffset);
        }

        if (content) {
          //Add the remaining content...
          newEntry.add(content, contentCFI, sentenceOffset);
          this.sentenceEntries.push(newEntry);
        }

        this.trim();
        return;
      };
      
      /**@public Add a line break*/
      PrefixManager.prototype.addLineBreak = function (contentCFI, offset) {
        if (!this.sentenceEntries.length) {
          return this;
        }

        this.getLastEntry().add(' ', contentCFI, offset).end();
        this.trim();
        
        return this;
      };

      /**@public Gets the entry in offset-map that contains the buffer character at given offset*/
      PrefixManager.prototype.getEntryForOffset = function (offset) {
        var offsetFromStart = 0,
          result;

        /**TODO: Should we instead do a reverse-traversal? Seems more complicated, marginal benefit*/
        offset = (offset < 0) ? this.getBuffer().length + offset : offset;

        this.sentenceEntries.some(function (sentence, index, map) {
          if (index >= map.length - 1
            || (offsetFromStart <= offset && offsetFromStart + sentence.getLength() > offset)) {
            result = sentence.getEntryForOffset(offset - offsetFromStart);
            return true;
          }

          offsetFromStart += sentence.getLength();
          return false;
        });

        return result;
      };
      /**
       * END PREFIX MANAGER SECTION
       */
      var readTillSentenceEnd = function (textNode, textOffset, cfi) {
        var startingOffset = textOffset,
          newIterator,
          endsAtDelimiter = false,
          endsAtQuote = false,
          textSoFar = '';

        newIterator = new AdobeCfiIterator(textNode, cfi);
        newIterator.traverse(function (node, cfi) {
          /*jslint unparam:true*/
          var thisText,
            shouldContinue = true,
            regexResult;

          if (node.nodeType === Node.TEXT_NODE) {
            thisText = node.wholeText;

            if (startingOffset) {
              thisText = thisText.substr(startingOffset);
              startingOffset = 0;
            }

            //Check if start of this node completes the pattern...
            if (endsAtDelimiter) {
              if (thisText.length > 0 && /['"]{0,1}\s/g.test(thisText.substr(0,2))) {
                shouldContinue = false;
              }
            } else if (endsAtQuote) {
              if (thisText.length > 0 && /\s/g.test(thisText[0])) {
                shouldContinue = false;
              }
            }

            if (shouldContinue && thisText.length) {
              regexResult = PrefixManager.sentenceEndRegex().exec(thisText);
              if (regexResult) {
                textSoFar += thisText.substr(0, regexResult.index + regexResult[0].length);
                shouldContinue = false;
              } else {
                textSoFar += thisText;
                endsAtDelimiter = /[.?!]/.test(thisText[thisText.length - 1]);
                endsAtQuote = thisText.length > 1 && /[.?!]['"]/.test(thisText[thisText.length - 2]);
              }
            }
          }

          return shouldContinue;
        });

        return textSoFar;
      };

      var  matchArray = [],
        regex,
        matchPattern;

      var sentenceMatches = textToFind.match(PrefixManager.sentenceEndRegex());
      var sentences = sentenceMatches ? sentenceMatches.length + 1 : 1;
      var prefixManager = new PrefixManager(textToFind.length - 1, sentences);

      /**First, escape all string combinations that could be interpreted as regex symbols
       * Then, replace all white spaces with the multiple white space pattern matcher (\s+)
       * This is a workaround for Bug #3832100 - the HTML renderer strips out whitespaces
       * and we need to do the same to provide intuitive results to the user.
       * Finally, create a regex with this pattern and the case insensitive flag
       */
      matchPattern = Utility.escapeRegexString(textToFind);
      matchPattern = matchPattern.replace(/\s+/g, '\\s+');
      regex = new RegExp(matchPattern, 'i');

      iterator.traverse(function(node, thisCfi, thisText, startOffset, endOffset) {
      try {
        /*jslint unparam:true*/
        var contents = thisText,
          index = -1,
          indexOffset = 0,
          matchLength,
          nextIndex,
          regexResult,
          shouldContinue = true,
          startCFI,
          endCFI,
          matchContext,
          indexInContext,
          sanitizedContextInfo,
          initialPrefix,
          matchPrefixEntry;

        //If we have an offset, the prefix does not immediately precede this text...
        if (startOffset) {
          prefixManager.empty();
        }
        indexOffset = startOffset;

        if (!contents.length) {
          return true;
        }

        if (!contents.trim().length) {
          prefixManager.addLineBreak(thisCfi, indexOffset);
          return true;
        }

        initialPrefix = prefixManager.minPrefix();
        contents = initialPrefix + contents;

        regexResult = contents.match(regex);
        while (shouldContinue && regexResult) {
          /**index and nextIndex are in combined string space*/
          index = regexResult.index;
          nextIndex = index + 1;
          matchLength = regexResult[0].length;
          
          if (index >= initialPrefix.length) {
            //Push in characters upto this match and retrieve entry...
            prefixManager.add(contents.substring(initialPrefix.length, nextIndex), thisCfi, indexOffset);
            matchPrefixEntry = prefixManager.getEntryForOffset(-1);
          } else {
            //Prefix already pushed - retrieve relevant entry...
            matchPrefixEntry = prefixManager.getEntryForOffset(index - initialPrefix.length);
          }

          /**Move to the first unmatched character
           * Remove the prefix since it's job is done*/
          contents = contents.substr(nextIndex);
          indexOffset += nextIndex - initialPrefix.length;
          initialPrefix = "";

          /**Construct details of this match using prefix manager
           * CFI highlighting is one edge inclusive => [index, index + matchLength)*/
          startCFI = matchPrefixEntry.cfi + ":" + matchPrefixEntry.offset;
          endCFI = thisCfi + ":" + (indexOffset + matchLength - 1);
          matchContext = prefixManager.getBuffer();
          indexInContext = matchContext.length - 1;
          // matchContext += contents.substr(0, matchLength - 1);
          matchContext += readTillSentenceEnd(node, indexOffset, thisCfi);

          /**Create a new object for match every time. (Closure)
           * Otherwise each entry overwrites the previous one's data*/
          sanitizedContextInfo = Utility.stripWhiteSpacesAndAdjustOffset(matchContext, indexInContext);
          matchArray.push({
            startCFI: startCFI,
            endCFI: endCFI,
            context: sanitizedContextInfo.text,
            index: sanitizedContextInfo.index,
            length: textToFind.replace(/\s+/g, ' ').length //Shouldn't this be regexResult[0].replace()?
          });
          if (!searchMultiple) {
            shouldContinue = false;
            break;
          }

          regexResult = contents.match(regex);
        }

        //Add remaining (unmatched) characters...
        prefixManager.add(contents.substr(initialPrefix.length), thisCfi, indexOffset);

        return shouldContinue;
      } catch (error) {
        console.error('DEBUG_ONLY: ERROR IN PREFIX STUFF!');
      }
      });

      return matchArray;
    },

    /**NOTES
     * See findInRange() */
    findInReverseRange: function (textToFind, iterator, searchMultiple) {
      var suffix = '',
        lastCfiInfo = {
          CFI: '',
          offset: 0
        },
        contextLength = 20, //characters after match...
        match = {
          startCFI: '',
          endCFI: '',
          context: ''
        },
        suffixLength = textToFind.length - 1,
        matchArray = [],
        regex,
        matchPattern;

      /**See findInRange()*/
      matchPattern = Utility.escapeRegexString(textToFind);
      matchPattern = matchPattern.replace(/\s+/g, '\\s+');
      regex = new RegExp(matchPattern, 'ig');

      iterator.rTraverse(function(node, thisCfi, text, startOffset, endOffset) {
        /*jslint unparam: true*/
        var contents = text,
          matchIndex = -1,
          matchLength,
          regexResult,
          shouldContinue = true,
          matchOffset,
          regexResultIndex,
          matchText;

        try {
          //If we have an offset, the suffix does not immediately precede this text...
          if (endOffset) {
            suffix = "";
            lastCfiInfo.CFI = "";
            lastCfiInfo.offset = 0;
          } else {
            contents = contents + suffix;
            endOffset = endOffset + suffix.length;
            match.startCFI = lastCfiInfo.CFI;
          }

          if (contents.length) {     
            regexResult = contents.match(regex);
            regexResultIndex = 0;

            if (regexResult) {
              regexResultIndex = regexResult.length;
            }
            while (shouldContinue && regexResultIndex > 0) {
              --regexResultIndex;
              /**matchIndex is index in current substring
               * matchOffset is index in complete content string*/
              matchText = regexResult[regexResultIndex];
              matchIndex = contents.lastIndexOf(matchText);
              matchLength = matchText.length;

              //This is a sanity check...
              if (matchIndex >= 0) {
                matchOffset = matchIndex + matchLength;

                if (matchOffset > text.length) {
                  //This match ends in the suffix...
                  match.endCFI = lastCfiInfo.CFI + ':' + (matchOffset - text.length + lastCfiInfo.offset);
                  match.context = contents.substring(matchIndex, -contextLength);
                } else {
                  /**This match ends in the current text...
                   * Note: When offset = text.length, we return an OutOfBounds CFI [character@currentnode.length]
                   * This is OK because endCFI is taken as exclusive. The alternative (returning
                   * lastCfiInfo.cfi + offset 0) causes excess highlighting issues since selection now
                   * extends to the start of the previous node, NOT the end of this one
                   */
                  match.endCFI = thisCfi + ':' + (startOffset + matchOffset);
                  match.context = text.substring(matchIndex - contextLength, matchIndex + contextLength);
                }
                
                /**By design, the match always starts in this node...
                 * Otherwise, it would have been found when matching the suffix itself*/
                match.startCFI = thisCfi + ':' + (startOffset + matchOffset - matchLength);

                //DEBUG ONLY
                match.realCFI = AdobeRDMHelper.generateTextElementCFI(node, startOffset + matchOffset - matchLength);
                
                matchArray.push(match);
                if (!searchMultiple) {
                  shouldContinue = false;
                  break;
                }

                //Move to the next match...
                contents = contents.substr(0, matchIndex);
              }
            }
          }

          //Update the suffix for the next match...
          suffix = contents.substr(0, suffixLength);
          lastCfiInfo.CFI = thisCfi;
          lastCfiInfo.offset = startOffset;
          if (lastCfiInfo.offset < 0) {
            lastCfiInfo.offset = 0;
          }
        } catch (error) {
          console.log(error);
        }

        return shouldContinue;
      });

      return matchArray;
    },

    // This method is added to workaround the IE inconsistency with the standard for "document.createTreeWalker()" method
    createTreeWalker: function(documentObj, rootNode, whatToShow, filter, entityReferenceExpansion){
      if(ReadiumSDK._isIEBrowser)
      {
        return documentObj.createTreeWalker(rootNode, whatToShow, filter.acceptNode, entityReferenceExpansion?true:false);
      }
      return documentObj.createTreeWalker(rootNode, whatToShow, filter, entityReferenceExpansion?true:false);
    }
  };
}());
// FILE END: AdobeRDMHelper.js

// FILE START: AdobeCfiIterator.js
/*jslint indent: 2*/
(function () {
  'use strict';

  /*global $, Node*/

  /**@constructor @private
   * Handles traversal of one level, from firstChild to lastChild
   * Moves in increments of CFI code, i.e each increment will definitely change CFI (see Adjacent Text Problem)
   * @param {String, Number} cfiIndex Cfi string for DOM corresponding to this stack state
   * NOTE Consider removing the parentNode and traversing purely horizontally through 'nextSibling'
   * NOTE The complicated _textRunSpan and _textChildrenSeen may be unnecessary since we are now using _otherChildrenSeen for CFI
   */
  var SingleLevelIterator = function (node, startChild, ignore) {
    if (!node) {
      throw new Error('AdobeCfiIterator: Invalid node for traversal');
    }
    startChild = startChild || node.firstChild;

    /**@public {Node} HTML Node context for this iteration*/
    this.parentNode = node;

    /**@public {String,Number} Cfi corresponding to current pointer state*/
    this.currentChildCfi = undefined;

    /**@private {Number}*/
    this._childIndex = 0;

    /**@private {Number}*/
    this._textChildrenSeen = 0; //This is currently unused in CFI calculation...

    /**@private {Number}*/
    this._otherChildrenSeen = 0;

    /**@private {Number}*/
    this._textRunSpan = 0;

    /**@private {Number}*/
    this._ignoreCount = 0;

    ignore = ignore || {};
    this._ignore = {
      ids: ignore.idList || [],
      classes: ignore.classList || [],
      elements: ignore.elementList || []
    };

    /**At any point, the iterator has a valid CFI 
     * string it contributes from this level*/
    this._setParent(node);
    this._initializeState(startChild);
  };

  /**@public Indicates if this node has a nextCFISibling*/
  SingleLevelIterator.prototype.canIncrement = function () {
    var jump = this._textRunSpan + this._ignoreCount;
    if (jump === 0) {
      ++jump;
    }
    return (this._childIndex + jump) < this.parentNode.childNodes.length;
  };

  /**@public Indicates if this node has a previousCFISibling*/
  SingleLevelIterator.prototype.canDecrement = function () {
    return this._childIndex > 0;
  };

  /**@private @param{Node} DOM Node to use as parent context*/
  SingleLevelIterator.prototype._setParent = function (node) {
    var eachChild;

    this.parentNode = node;

    this._textChildrenSeen = 0;
    this._otherChildrenSeen = 0;
    this._childIndex = 0;

    this._textRunSpan = this._ignoreCount = 0;
    while (this._textRunSpan + this._ignoreCount < this.parentNode.childNodes.length) {
      eachChild = this.parentNode.childNodes[this._textRunSpan + this._ignoreCount];
      if (eachChild.nodeType === Node.TEXT_NODE) {
        ++this._textRunSpan;
      } else if (eachChild.nodeType === Node.COMMENT_NODE) {
        ++this._ignoreCount;
      } else {
        //Non-text, non-comment node: Break!
        break;
      }
    }
  };

  /**@private @param{Node} DOM Node to move pointer to*/
  SingleLevelIterator.prototype._initializeState = function (childToFind) {
    var childNodes = this.parentNode.childNodes,
      childCount = childNodes.length,
      eachChild,
      foundChild = false;

    while (this._childIndex < childCount && !foundChild) {
      eachChild = childNodes[this._childIndex];
      if (eachChild === childToFind) {
        foundChild = true;
        break;
      }
      this._countChild(eachChild);
    }

    if (childToFind && !foundChild) {
      throw new Error('AdobeCfiIterator: Invalid parent-child relationship');
    }

    //Make the childCfi even if we wanted to find the firstChild and _countChild was never called
    this.currentChildCfi = this._currentCfi();
  };

  /**@public @return{Node} DOM Node pointed to*/
  SingleLevelIterator.prototype.currentChild = function () {
    return this.parentNode.childNodes[this._childIndex];
  };

  /**@private Compute cfi for this state*/
  SingleLevelIterator.prototype._currentCfi = function () {
    var child = null,
      cfi = '';

    if (this._textRunSpan) {
      //Compute CFI for text node...
      // cfi = (this._textChildrenSeen * 2) + 1;
      cfi = (this._otherChildrenSeen * 2) + 1;
    } else {
      cfi = (this._otherChildrenSeen + 1) * 2;
      child = this.currentChild();
      if (child.id) {
        cfi += '[' + child.id + ']';
      }
    }
    cfi.toString();
    return cfi;
  };

  /**@private @param {Node} Moves the pointer to next CFI node in this level*/
  SingleLevelIterator.prototype._countChild = function () {
    var eachChild;

    /*Update values to include current*/
    if (this._textRunSpan) {
      ++this._textChildrenSeen;
      this._childIndex += this._textRunSpan;
    } else {
      ++this._otherChildrenSeen;
      ++this._childIndex;
    }

    /*And move to next*/
    this._textRunSpan = this._ignoreCount = 0;
    while (this._childIndex + this._textRunSpan + this._ignoreCount < this.parentNode.childNodes.length) {
      eachChild = this.parentNode.childNodes[this._childIndex + this._textRunSpan + this._ignoreCount];
      if (eachChild.nodeType === Node.TEXT_NODE) {
        ++this._textRunSpan;
      } else if (eachChild.nodeType === Node.COMMENT_NODE) {
        ++this._ignoreCount;
      } else {
        //Non-text, non-comment node: Break!
        break;
      }
    }
    
    this.currentChildCfi = this._currentCfi();
  };

  /**@private @param {Node} Moves the pointer to previous CFI node in this level*/
  SingleLevelIterator.prototype._uncountChild = function () {
    var eachChild;
    
    /**Move to previous*/
    --this._childIndex;
    this._textRunSpan = this._ignoreCount = 0;
    while (this._childIndex - this._textRunSpan - this._ignoreCount >= 0) {
      eachChild = this.parentNode.childNodes[this._childIndex - this._textRunSpan - this._ignoreCount];
      if (eachChild.nodeType === Node.TEXT_NODE) {
        ++this._textRunSpan;
      } else if (eachChild.nodeType === Node.COMMENT_NODE) {
        ++this._ignoreCount;
      } else {
        //Non-text, non-comment node: Break!
        break;
      }
    }
    this._childIndex -= (this._textRunSpan > 1) ? (this._ignoreCount + this._textRunSpan - 1) : this._ignoreCount;

    /*And update values to exclude it*/
    if (this._textRunSpan) {
      --this._textChildrenSeen;
    } else {
      --this._otherChildrenSeen;
    }

    this.currentChildCfi = this._currentCfi();
  };

  /**@public {Number} toChildIndex DOM-index to move to*/
  SingleLevelIterator.prototype.fastForward = function (toChildIndex) {
    var childNodes = this.parentNode.childNodes,
      childCount = childNodes.length,
      result = {
        cfi: 0,
        node: null
      },
      textOffset;

    /**Handle trivial cases*/
    if (this._childIndex > toChildIndex) {
      throw new Error('AdobeCfiIterator: Invalid fast forward operation');
    }

    if (toChildIndex >= childCount) {
      throw new Error('AdobeCfiIterator: Out of bounds');
    }

    while (this._childIndex < toChildIndex) {
      //Text offset only comes into the picture for currentNode being textNode
      textOffset = (this._textRunSpan > 0) ? (this._textRunSpan - 1) : 0;
      if (toChildIndex > this._childIndex + textOffset) {
        this._countChild();
      } else {
        break;
      }
    }

    result.node = childNodes[this._childIndex];
    result.cfi = this.currentChildCfi;
    return result;
  };

  /**@public {Number} toChildIndex DOM-index to move to*/
  SingleLevelIterator.prototype.rewind = function (toChildIndex) {
    var childNodes = this.parentNode.childNodes,
      result = {
        cfi: 0,
        node: null
      };

    /**Handle trivial cases*/
    if (this._childIndex < toChildIndex) {
      throw new Error('AdobeCfiIterator: Invalid rewind operation');
    }

    if (toChildIndex < 0) {
      throw new Error('AdobeCfiIterator: Out of bounds');
    }

    while (this._childIndex > toChildIndex) {
      this._uncountChild();
    }

    result.node = childNodes[this._childIndex];
    result.cfi = this.currentChildCfi;
    return result;
  };

  /**@public {Number} toChildIndex DOM-index to move to*/
  SingleLevelIterator.prototype.goto = function (toChildIndex) {
    if (this._childIndex > toChildIndex) {
      return this.rewind(toChildIndex);
    }
    if (this._childIndex < toChildIndex) {
      return this.fastForward(toChildIndex);
    }
  };

  /**@public Go to the node corresponding to the next CFI*/
  SingleLevelIterator.prototype.increment = function () {
    var nextOffset = (this._textRunSpan > 1) ? this._textRunSpan : 1;
    this.fastForward(this._childIndex + nextOffset);
  };

  /**@public Go to the node corresponding to the previous CFI*/
  SingleLevelIterator.prototype.decrement = function () {
    this.rewind(this._childIndex - 1);
  };

  /**@constructor Handles when to traverse horizontally and when vertically
   * @param {Node} startNode Node that determines iterator's position
   * @param {String} startCFI Partial CFI that corresponds to this node
   * NOTE: Iterator will not traverse above the root of this partial CFI
   */
  var AdobeCfiIterator = function (startNode, startCFI, existingIterator) {
    /**@private {Array<SingleLevelIterator>} Stack info for depth-first traversal*/
    this._levels = [];

    /**@private {Array<SingleLevelIterator>} Excess, unvisited nodes on the path to startNode*/
    this._cfiBuffer = [];

    /**@private {String} Cfi index of root of current subtree
     * Changed only when new levels taken from buffer*/
    this._rootCfi = '';

    /**Clone an existing iterator*/
    if (existingIterator) {
      this._levels = existingIterator._levels.slice();
      this._cfiBuffer = existingIterator._cfiBuffer.slice();
      this._rootCfi = existingIterator._rootCfi;
      return this;
    }

    //This check seems to fail on webkit. NodeConstructor !== NodePrototype !== Node
    //if (!startNode || !(startNode instanceof Node)) {
    if (!startNode) {
      throw new Error('AdobeCfiIterator: Invalid start node');
    }
    if (!startCFI) {
      throw new Error('AdobeCfiIterator: No CFI passed');
    }

    /**Any iterator has a valid startNode and a CFI corresponding to that node
     * The iterator's domain is the root of the CFI tree given 
     * CFI Buffer, root and levels give the entire partial CFI at any given time */
    this._initialize(startNode, startCFI);
  };

  /**@private @static*/
  AdobeCfiIterator._stripTemporalSpatialSymbols = function (cfiString) {
    /*http://www.idpf.org/epub/linking/cfi/epub-cfi.html#sec-path-res*/
    var symbolIndex = -1,
      nextSlashIndex = -1,
      // len = cfiString.length,
      result = '';

    while (cfiString.length) {
      symbolIndex = cfiString.search(/[@~]/);
      if (symbolIndex > -1) {
        result += cfiString.substring(0, symbolIndex);
        nextSlashIndex = cfiString.indexOf('/', symbolIndex + 1);
        if (nextSlashIndex === -1) {
          //Strip out the entire remaining string...
          cfiString = '';
        } else {
          //Add Slash and move the pointer to next character...
          result += '/';
          cfiString = cfiString.substring(nextSlashIndex + 1);
        }
      } else {
        result += cfiString;
        cfiString = '';
      }
    }
    return result;
  };

  /**@private @static*/
  AdobeCfiIterator._createListFromCfiString = function (cfiString) {
    /*Make a partial state info stack from CFI*/
    var cfi = cfiString.split('/');
    if (cfiString === '/') {
      //'/' -> ['','']
      cfi.pop();
    }
    return cfi;
  };

  /**@public @static*/
  AdobeCfiIterator.getNumberFromString = function (cfiNodeStr) {
    //Match for digits until a non digit or a null-char
    cfiNodeStr = cfiNodeStr ? cfiNodeStr.toString() : "";
    var matches = cfiNodeStr.match(/\d+/),
      result = 0;

    if (matches && matches.length) {
      result = parseInt(matches[0], 10);
      if (typeof result !== 'number') {
        result = 0;
      }
    }
    return result;
  };

  /**@public @static*/
  AdobeCfiIterator.normalizeCFI = function (cfiString) {
    //Add any steps to understand standard CFI strings here...
    var normalString = AdobeCfiIterator._stripTemporalSpatialSymbols(cfiString);
    return normalString;
  };

  /**@public @static*/
  AdobeCfiIterator.CFIRelationships = {
    EqualTo: 0,
    GreaterThan: 1,
    LesserThan: 2,
    ParentOf: 3,
    ChildOf: 4
  };

  /**@public @static Assumes well formed cfi string, does not support ranges*/
  AdobeCfiIterator.stripOffsetFromCfiList = function (cfi) {
    var offset = 0,
      cfiStr,
      split;
    if (cfi && cfi.length) {
      //Support a list of strings as well as single string...
      cfiStr = cfi[cfi.length - 1];
      split = cfiStr.split(':');
      if (split.length > 1) {
        cfiStr = split[0];
        offset = parseInt(split[1], 10);
        if (typeof offset !== 'number') {
          offset = 0;
        }
        cfi[cfi.length - 1] = cfiStr;
      }
    }
    return offset;
  };
  
  /**@public @static
   * Similar to lexical compare, returns diff of first non-matching node
   * @param {String} cfiOne First normalized CFI String
   * @param {String} cfiTwo Second normalized CFI String
   * @return{AdobeCfiIterator.CFIRelationships} Relationship of one with two
   */
  AdobeCfiIterator.cfiCompare = function (cfiOne, cfiTwo) {
    var diff = 0,
      listOne = AdobeCfiIterator._createListFromCfiString(cfiOne),
      listTwo = AdobeCfiIterator._createListFromCfiString(cfiTwo),
      lengthOne = listOne.length,
      lengthTwo = listTwo.length,
      partOne,
      partTwo,
      offsetOne,
      offsetTwo,
      i,
      result = AdobeCfiIterator.CFIRelationships.EqualTo;

    for (i = 0; i < lengthOne && i < lengthTwo; ++i) {
      partOne = AdobeCfiIterator.getNumberFromString(listOne[i]);
      partTwo = AdobeCfiIterator.getNumberFromString(listTwo[i]);
      diff = partOne - partTwo;

      if (diff > 0) {
        result = AdobeCfiIterator.CFIRelationships.GreaterThan;
        break;
      }
      if (diff < 0) {
        result = AdobeCfiIterator.CFIRelationships.LesserThan;
        break;
      }
    }
    if (diff === 0) {
      if (lengthOne === lengthTwo) {
        //The last elements of each CFI string may have a text-offset - strip that into a new CFI part...
        offsetOne = AdobeCfiIterator.stripOffsetFromCfiList(listOne);
        offsetTwo = AdobeCfiIterator.stripOffsetFromCfiList(listTwo);
        if (offsetOne < offsetTwo) {
          result = AdobeCfiIterator.CFIRelationships.LesserThan;
        } else if (offsetOne > offsetTwo) {
          result = AdobeCfiIterator.CFIRelationships.GreaterThan;
        }
      } else if (lengthOne > lengthTwo){
        result = AdobeCfiIterator.CFIRelationships.ChildOf;
      } else {
        result = AdobeCfiIterator.CFIRelationships.ParentOf;
      }
    } 
    return result;
  };

  /**@private*/
  AdobeCfiIterator.prototype._initialize = function (startNode, startCFI) {
    var startLevel,
      childNode;

    if (!startNode) {
      throw new Error('AdobeCfiIterator: Invalid start node');
    }

    this._cfiBuffer = AdobeCfiIterator._createListFromCfiString(startCFI);

    //Make startNode the currentNode in a level...
    if (this._cfiBuffer.length) {
      this._cfiBuffer.pop();
    }
    childNode = startNode;
    startNode = startNode.parentNode;

    if (!startNode || !startNode.firstChild) {
      throw new Error('AdobeCfiIterator: Unexpected DOM structure - text node with text parent');
    }

    if (this._cfiBuffer.length) {
      this._rootCfi = this._cfiBuffer.pop();
    }
    startLevel = new SingleLevelIterator(startNode, childNode);
    this._levels.push(startLevel);
  };

  /**@private*/
  AdobeCfiIterator.prototype._getCurrentLevel = function () {
    return this._levels[this._levels.length - 1];
  };

  /**@private*/
  AdobeCfiIterator.prototype._getCurrentNode = function () {
    var node = null,
      level = this._getCurrentLevel();
    if (level) {
      node = level.currentChild();
    }
    return node;
  };

  /**@private Move one step up the CFI-DOM tree*/
  AdobeCfiIterator.prototype._stepUp = function () {
    var oldLevel = this._levels.pop(),
      currentChild = oldLevel.parentNode,
      currentLevel;

    if (!this._levels.length) {
      //Check if we have some unvisited parents in buffer...
      if (this._cfiBuffer.length) {
        this._rootCfi = this._cfiBuffer.pop();
        currentLevel = new SingleLevelIterator(currentChild.parentNode, currentChild);
        this._levels.push(currentLevel);
      }
    }

    return this;
  };

  AdobeCfiIterator.prototype._stepDown = function (childIndex) {
    var currentLevel = this._getCurrentLevel(),
      newLevel;

    newLevel = new SingleLevelIterator(currentLevel.currentChild());

    if (newLevel.parentNode) {
      newLevel.fastForward(childIndex);
      this._levels.push(newLevel);
    }

    return this;
  };

  /**@public Utility function to clone iterator*/
  AdobeCfiIterator.prototype.clone = function () {
    return new AdobeCfiIterator(null, null, this);
  };

  /**@public Returns a new iterator to this parent*/
  AdobeCfiIterator.prototype.parent = function () {
    var result = this.clone();
    return result._stepUp();
  };

  /**@public @param{Number} childIndex*/
  AdobeCfiIterator.prototype.child = function (childIndex) {
    var result = this.clone();
    return result._stepDown(childIndex);
  };

  /**@public Returns iterator to next CFI element*/
  AdobeCfiIterator.prototype.next = function () {
    var result = this.clone();
    return result.increment();
  };

  /**@public Moves to the next CFI element*/
  AdobeCfiIterator.prototype.increment = function () {
    var currentLevel = this._getCurrentLevel();

    if (!this.isValid()) {
      throw new Error('AdobeCfiIterator: Invalid state!');
    }

    //Never push an element without children into a new Level...
    if (currentLevel.currentChild().childNodes.length > 0) {
      this._stepDown(0);
    } else if (currentLevel.canIncrement()) {
      currentLevel.increment();
    } else {
      do {
        this._stepUp();
        if (this.isValid()) {
          currentLevel = this._getCurrentLevel();
          if (currentLevel.canIncrement()) {
            currentLevel.increment();
            break;
          }
        }
      } while (this.isValid());
    }
    return this;
  };

  AdobeCfiIterator.prototype.previous = function () {
    var result = this.clone();
    return result.decrement();
  };

  AdobeCfiIterator.prototype.decrement = function () {
    var currentLevel = this._getCurrentLevel(),
      currentNode;

    if (!this.isValid()) {
      throw new Error('AdobeCfiIterator: Invalid state!');
    }

    //Check if we have a previous sibling of currentNode...
    if (currentLevel.canDecrement()) {
      currentLevel.decrement();
      currentNode = currentLevel.currentChild();
      //Keep moving to lastChild till we can't...
      while (currentNode.childNodes.length) {
        this._stepDown(currentNode.childNodes.length - 1);
        currentNode = this._getCurrentNode();
        
        //Sanity check
        if (!currentNode) {
          throw new Error('AdobeCfiIterator: Unexpected null child encountered');
        }
      }
    } else {
      //Move the parent...
      this._stepUp();
    }

    return this;
  };

  /**@public*/
  AdobeCfiIterator.prototype.isValid = function () {
    var areNodesAvailable = (this._levels.length || this._cfiBuffer.length),
      isCurrentHeadValid = this._getCurrentNode() !== null;

    return (areNodesAvailable && isCurrentHeadValid);
  };

  /**@public The CFI String corresponding to this point*/
  AdobeCfiIterator.prototype.getCfiString = function () {
    var cfi = this._cfiBuffer.join('/');

    if (this._rootCfi !== '') {
      cfi += '/' + this._rootCfi;
    }
    this._levels.filter(function (level, index, stateInfo) {
      /*jslint unparam: true*/
      cfi += '/' + level.currentChildCfi;
    });
    return cfi;
  };

  /**@public Utility to traverse the CFI DOM
   * @param {Function} callback Called with iterator, current DOM Node and CFI String
   * Return false from the callback to stop traversal
   * Return value is last DOM Node + CFI String traversed*/
  AdobeCfiIterator.prototype.traverse = function (callback) {
    var shouldContinue = true,
      result = {};
    while (this.isValid() && shouldContinue !== false) {
      result.node = this._getCurrentNode();
      result.cfi = this.getCfiString();

      shouldContinue = callback.call(this, result.node, result.cfi);
      this.increment();
    }
    return result;
  };

  /**@public Debug Utility
   * NOTE Remove this*/
  AdobeCfiIterator.prototype.nameThyself = function () {
    var str = '',
      cfiStr = this.getCfiString(),
      currentNode = this._getCurrentNode();

    if (currentNode.nodeType === Node.TEXT_NODE) {
      str += '"' + currentNode.nodeValue.trim() + '"';
    }
    str += '@[' + cfiStr + ']';
    return str;
  };

  /**@constructor
   * @param {Node} startNode See AdobeCfiIterator
   * @param {String} startCFI See AdobeCfiIterator
   * @param {String} lowerBoundCFI Partial CFI that corresponds to the end node
   * @param {String} upperBoundCFI Partial CFI that corresponds to the end node
   */
  var AdobeCfiBoundedIterator = function (startNode, startCFI, lowerBoundCFI, upperBoundCFI) {
    /*jslint unparam: true*/
    AdobeCfiIterator.prototype.constructor.call(this, startNode, startCFI);
    
    lowerBoundCFI = lowerBoundCFI || startCFI;
    upperBoundCFI = upperBoundCFI || '/';
    
    this._upperBoundInfo = {
      /**@private {Array<String>} List of CFI indices denoting a bound*/
      path: AdobeCfiIterator._createListFromCfiString(upperBoundCFI),

      offset: 0,
      
      /**@private {Number} Index upto which current stack matches bound
       * Starts before the common root*/
      matchIndex: -1,
      
      /**@private {Boolean}*/
      atBound: false,

      /**@private {Boolean}*/
      crossed: false
    };
    this._upperBoundInfo.offset = AdobeCfiIterator.stripOffsetFromCfiList(this._upperBoundInfo.path);

    this._lowerBoundInfo = {
      /**@private {Array<String>} List of CFI indices denoting a bound*/
      path: AdobeCfiIterator._createListFromCfiString(lowerBoundCFI),
      
      offset: 0,

      /**@private {Number} Index upto which current stack matches bound
       * Starts before the common root*/
      matchIndex: -1,
      
      /**@private {Boolean}*/
      atBound: false,

      /**@private {Boolean}*/
      crossed: false
    };
    this._lowerBoundInfo.offset = AdobeCfiIterator.stripOffsetFromCfiList(this._lowerBoundInfo.path);

    /**NOTE Known bug when starting state itself is invalid for lowerBounds
     * Should be fixed if ensure bounds equals check is changed to less/greater*/
    /**Expected to match the common root and set matchIndex to 0 at least*/
    this._ensureBounds(this._lowerBoundInfo, true);
    this._ensureBounds(this._upperBoundInfo);
  };

  AdobeCfiBoundedIterator.prototype = Object.create(AdobeCfiIterator.prototype);
  AdobeCfiBoundedIterator.prototype.constructor = AdobeCfiBoundedIterator;

  /**@override*/
  AdobeCfiBoundedIterator.prototype.isValid = function () {
    return (!this._upperBoundInfo.crossed && !this._lowerBoundInfo.crossed
      && AdobeCfiIterator.prototype.isValid.apply(this));
  };

  /**@override*/
  AdobeCfiBoundedIterator.prototype.increment = function () {
    /*jslint unparam: true*/
    AdobeCfiIterator.prototype.increment.apply(this, arguments);
    if (this._lowerBoundInfo.atBound) {
      this._lowerBoundInfo.atBound = false;
    }
    this._ensureBounds(this._upperBoundInfo, false);
    return null;
  };

  /**@override*/
  AdobeCfiBoundedIterator.prototype.decrement = function () {
    /*jslint unparam: true*/
    AdobeCfiIterator.prototype.decrement.apply(this, arguments);
    if (this._upperBoundInfo.atBound) {
      this._upperBoundInfo.atBound = false;
    }
    this._ensureBounds(this._lowerBoundInfo, true);
    return null;
  };

  /**@public @static Utility to check if two CFI tags are equal
   * @param {Number,String} cfiOne
   * @param {Number,String} cfiTwo
   */
  AdobeCfiBoundedIterator.cfiStringsAreEqual = function (cfiOne, cfiTwo) {
    var first = AdobeCfiIterator.getNumberFromString(cfiOne),
      second = AdobeCfiIterator.getNumberFromString(cfiTwo);

    return first === second;
  };

  /**@private Crux of the Bound logic
   * Matches against bounds at every increment and detects
   * when an unlawful increment, i.e, that hits already 
   * matched element, is reached
   * NOTE Longish function because of the 3-part check. Refactor.
   * @param {Object} boundInfo Bounds information passed by reference
   */
  AdobeCfiBoundedIterator.prototype._ensureBoundsOld = function (boundInfo, ensureGreater) {
    /*jslint unparam:true*/
    var levelLength = this._levels.length,
      boundLength = boundInfo.path.length,
      bufferLength = this._cfiBuffer.length,
      nextIndex,
      offset = 0,
      isMatch = true,
      levelIndexChanged,
      boundCfi,
      nodeCfi;

    /**As against the buffer, we have buffer + root + levels...
     * In index terms that's buffer.length + 1 + level.length - 1 */
    levelIndexChanged = levelLength + bufferLength;

    /**levelLength - 1 is the index last updated
     * Once we match an index, the stack operations can *never* go above that
     * i.e, Once 2/4/4 is matched, 2/4/5, 2/5, 3 are all illegal
     */
    if (boundInfo.matchIndex > levelIndexChanged) {
      boundInfo.crossed = true;
      return;
    } 
    if (boundInfo.matchIndex === levelIndexChanged) {
      //Ensure the change invalidates the current node...
      boundCfi = AdobeCfiIterator.getNumberFromString(boundInfo.path[levelIndexChanged]);
      nodeCfi = AdobeCfiIterator.getNumberFromString(this._getCurrentLevel().currentChildCfi);
      if (boundCfi !== nodeCfi) {
        boundInfo.crossed = true;
        return;
      }
    }

    //We'll use nextIndex as an alias to make it easier to read...
    nextIndex = boundInfo.matchIndex + 1;

    //First, check against the buffer...
    while (isMatch && nextIndex < boundLength && nextIndex < bufferLength) {
      if (!AdobeCfiBoundedIterator.cfiStringsAreEqual(this._cfiBuffer[nextIndex], boundInfo.path[nextIndex])) {
        isMatch = false;
      } else {
        ++nextIndex;
      }
    }

    //Now, check against the root node...
    offset = bufferLength;
    while (isMatch && nextIndex < boundLength && nextIndex < bufferLength + 1) {
      if (!AdobeCfiBoundedIterator.cfiStringsAreEqual(this._rootCfi, boundInfo.path[nextIndex])) {
        isMatch = false;
      } else {
        ++nextIndex;
      }
    }

    //Finally, check against our stacked levels...
    ++offset;
    while (isMatch && nextIndex < boundLength && nextIndex - offset < levelLength) {
      if (!AdobeCfiBoundedIterator.cfiStringsAreEqual(this._levels[nextIndex - offset].currentChildCfi, boundInfo.path[nextIndex])) {
        isMatch = false;
      } else {
        ++nextIndex;
      }
    }

    boundInfo.matchIndex = nextIndex - 1;
  };

  AdobeCfiBoundedIterator.prototype._ensureBounds = function (boundInfo, ensureGreater) {
    var boundCFI = boundInfo.path.join('/'),
      currentCfi = this.getCfiString(),
      relationship;

    relationship = AdobeCfiIterator.cfiCompare(currentCfi, boundCFI);

    boundInfo.atBound = boundInfo.crossed = false;
    if (relationship === AdobeCfiIterator.CFIRelationships.EqualTo) {
      boundInfo.atBound = true;
    } else if (ensureGreater && (relationship === AdobeCfiIterator.CFIRelationships.LesserThan
          || relationship === AdobeCfiIterator.CFIRelationships.ParentOf)) {
      boundInfo.crossed = true;
    } else if (!ensureGreater && (relationship === AdobeCfiIterator.CFIRelationships.GreaterThan
          || relationship === AdobeCfiIterator.CFIRelationships.ChildOf)) {
      /**Legacy behavior of iterator is to be valid until the sub tree of the bound is exhausted
       * i.e, /4/4/2/4/2/1 --> /4 is a valid condition and executes till entire subtree /4 is traversed*/
      /* This behavior is changed because of bug #3831658 related to double counted search list results */
      boundInfo.crossed = true;
    }
    return;
  };

  /**@constructor
   * @param {Node} startNode See AdobeCfiIterator
   * @param {String} startCFI See AdobeCfiIterator
   * @param {String} endCFI Partial CFI that corresponds to the end node
   */
  var AdobeCfiTextIterator = function (startNode, startCFI, lowerBoundCFI, upperBoundCFI) {
    /*jslint unparam: true*/
    AdobeCfiBoundedIterator.prototype.constructor.apply(this, arguments);
    while (this.isValid() && this._getCurrentNode().nodeType !== Node.TEXT_NODE) {
      AdobeCfiBoundedIterator.prototype.increment.apply(this, arguments);
    }

    /**Bug#3833532
     * If we were unable to get a valid text node moving forwards, try moving backwards... */
    if (!this.isValid()) {
      AdobeCfiBoundedIterator.prototype.constructor.apply(this, arguments);
      while (this.isValid() && this._getCurrentNode().nodeType !== Node.TEXT_NODE) {
        AdobeCfiBoundedIterator.prototype.decrement.apply(this, arguments);
      }
    }

    this._nodeOffset = AdobeCfiIterator.stripOffsetFromCfiList([startCFI.substr(startCFI.lastIndexOf('/') + 1)]);
    if (this._nodeOffset < 0) {
      this._nodeOffset = 0;
    }
    /**Note: By convention, start node offset is inclusive and other offsets are exclusive*/
    
    /**@private Is this the first text node we're traversing?*/
    /**NOTE What happens when I give a non text node + offset*/
    this._isStartNode = true;
  };

  AdobeCfiTextIterator.prototype = Object.create(AdobeCfiBoundedIterator.prototype);
  AdobeCfiTextIterator.prototype.constructor = AdobeCfiTextIterator;

  /**@override*/
  AdobeCfiTextIterator.prototype.increment = function () {
    //Invalidate the first node flag...
    if (this._isStartNode) {
      this._isStartNode = false;
    }

    //Move to the next text node...
    do {
      AdobeCfiBoundedIterator.prototype.increment.apply(this, arguments);
    } while (this.isValid() && this._getCurrentNode().nodeType !== Node.TEXT_NODE);
    return this;
  };

  /**@override*/
  AdobeCfiTextIterator.prototype.decrement = function () {
    //Invalidate the first node flag...
    if (this._isStartNode) {
      this._isStartNode = false;
    }

    //Move to the next text node...
    do {
      AdobeCfiBoundedIterator.prototype.decrement.apply(this, arguments);
    } while (this.isValid() && this._getCurrentNode().nodeType !== Node.TEXT_NODE);
    return this;
  };

  /**@public*/
  AdobeCfiTextIterator.prototype.getStartOffset = function () {
    return this._nodeOffset;
  };

  /**@override*/
  AdobeCfiTextIterator.prototype.traverse = function (callback) {
    var shouldContinue = true,
      result = {},
      adjustedOffset;
    while (this.isValid() && shouldContinue !== false) {
      result.node = this._getCurrentNode();
      result.cfi = this.getCfiString();
      result.text = result.node.wholeText;
      result.startOffset = 0;
      result.endOffset = 0;

      if (this._isStartNode && this._nodeOffset) {
        //If offset exists for this node (start node) trim the text...
        result.text = result.text.substr(this._nodeOffset);
        result.startOffset = this._nodeOffset;
      }

      if (this._upperBoundInfo.atBound) {
        adjustedOffset = this._upperBoundInfo.offset;
        //If this is also the first node, account for the trimming above...
        if (this._isStartNode) {
          adjustedOffset -= this._nodeOffset;
        }
        //If end bound offset exists, trim the text...
        if (adjustedOffset > -1) {
          result.text = result.text.substr(0, adjustedOffset);  
          result.endOffset = this._upperBoundInfo.offset;
        }
      }

      shouldContinue = callback.call(this, result.node, result.cfi, result.text, result.startOffset, result.endOffset);
      this.increment();
    }
    return result;
  };

  /**@override*/
  AdobeCfiTextIterator.prototype.rTraverse = function (callback) {
    var shouldContinue = true,
      result = {},
      adjustedOffset;

    while (this.isValid() && shouldContinue !== false) {
      result.node = this._getCurrentNode();
      result.cfi = this.getCfiString();
      result.text = result.node.wholeText;
      result.startOffset = 0;
      result.endOffset = 0;

      if (this._lowerBoundInfo.atBound && this._lowerBoundInfo.offset > 0) {
        //If offset exists for this node (first node) trim the text...
        result.text = result.text.substr(this._lowerBoundInfo.offset);
        result.startOffset = this._lowerBoundInfo.offset;
      }

      if (this._isStartNode) {
        //If this is also the first node, account for the trimming above...
        adjustedOffset = this._nodeOffset + 1 - result.startOffset;
        //If start bound offset exists, trim the text...
        if (adjustedOffset > -1) {
          result.text = result.text.substr(0, adjustedOffset);  
          result.endOffset = this._nodeOffset;
        }
      }

      shouldContinue = callback.call(this, result.node, result.cfi, result.text, result.startOffset, result.endOffset);
      this.decrement();
    }
    return result;
  };


  AdobeRDMHelper.ScreenInfoCache = {

    ReadingDir: {
      FORWARD: 0, //Set when there is a openPageNext() call
      BACKWARD: 1, //Set when there is a openPagePrevious() call
      RANDOM: 2 //Set when navigating to a random page
    },

    CurrentScreenInfo: {
      firstVisibleNode: undefined,
      lastVisibleNode: undefined,
      firstVisibleNodePercentHeight: 0,
      lastVisibleNodePercentHeight: 0,
      firstVisibleIdRef: undefined,
      lastVisibleIdRef: undefined,
      firstVisibleCFIPartial: undefined,
      lastVisibleCFIPartial: undefined  
    }
  };

  AdobeRDMHelper.ScreenInfoCache.CurrentScreenInfoState = {
      shouldRecalculateFirstVisible: true,
      shouldRecalculateLastVisible: true,
      readingDir: AdobeRDMHelper.ScreenInfoCache.ReadingDir.FORWARD, 
      screenInfo: AdobeRDMHelper.ScreenInfoCache.CurrentScreenInfo,
      isFixedLayout: false,
      isValid: function(){
        return !this.shouldRecalculateFirstVisible && !this.shouldRecalculateLastVisible;
      },

      invalidate: function(readingDir){
        this.shouldRecalculateFirstVisible = true;
        this.shouldRecalculateLastVisible = true;
        this.isFixedLayout = false;
        if(readingDir) {
          this.readingDir = readingDir;
        }

        if(this.readingDir === AdobeRDMHelper.ScreenInfoCache.ReadingDir.RANDOM)
        {
          this.screenInfo.firstVisibleNode = undefined;
          this.screenInfo.lastVisibleNode = undefined;
          this.screenInfo.firstVisibleNodePercentHeight = 0;
          this.screenInfo.lastVisibleNodePercentHeight = 0;
          this.screenInfo.firstVisibleIdRef = undefined;
          this.screenInfo.lastVisibleIdRef = undefined;
          this.screenInfo.firstVisibleCFIPartial = undefined;
          this.screenInfo.lastVisibleCFIPartial = undefined;
        }
      },

      invalidateFixedLayoutFirstVisibleNode: function(){
        this.isFixedLayout = true;
        this.shouldRecalculateFirstVisible = true;
        this.shouldRecalculateLastVisible = false;
        this.screenInfo.firstVisibleNode = undefined;
      },

      invalidateFixedLayoutLastVisibleNode: function(){
        this.isFixedLayout = true;
        this.shouldRecalculateFirstVisible = false;
        this.shouldRecalculateLastVisible = true;
        this.screenInfo.lastVisibleNode = undefined;
      }  
    };

    // Shortcut to CurrentScreenInfoState
    AdobeRDMHelper.ScreenInfoCache.CSIS = AdobeRDMHelper.ScreenInfoCache.CurrentScreenInfoState;

  /*global window*/
  window.AdobeCfiIterator = AdobeCfiIterator;
  window.AdobeCfiBoundedIterator = AdobeCfiBoundedIterator;
  window.AdobeCfiTextIterator = AdobeCfiTextIterator;

  return AdobeRDMHelper;
}());
// FILE END: AdobeCfiIterator.js
